#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2016 Antoine Le Gonidec <vv221@dotslashplay.it>
# SPDX-FileCopyrightText: © 2018 Jacek Szafarkiewicz
# SPDX-FileCopyrightText: © 2018 Phil Morrell
set -o errexit

###
# Unreal Tournament
# send your bug reports to vv221@dotslashplay.it
###

script_version=20251124.4

PLAYIT_COMPATIBILITY_LEVEL='2.33'

GAME_ID='unreal-tournament'
GAME_NAME='Unreal Tournament'

## This game is no longer available for sale from gog.com since 2022-12-23.
ARCHIVE_BASE_0_NAME='setup_ut_goty_2.0.0.5.exe'
ARCHIVE_BASE_0_MD5='0d25ec835648710a098aff7106187f38'
## Do not convert file paths to lowercase.
ARCHIVE_BASE_0_EXTRACTOR='innoextract'
ARCHIVE_BASE_0_EXTRACTOR_OPTIONS=' '
ARCHIVE_BASE_0_SIZE='640000'
ARCHIVE_BASE_0_VERSION='1.0-gog2.0.0.5'

ARCHIVE_REQUIRED_ENGINE_2_NAME='OldUnreal-UTPatch469e-Linux-amd64.tar.bz2'
ARCHIVE_REQUIRED_ENGINE_2_MD5='6ce309ddf5fb0d60c33be2f2ba92c71b'
ARCHIVE_REQUIRED_ENGINE_2_SIZE='191188'
ARCHIVE_REQUIRED_ENGINE_2_VERSION='469e-1'
ARCHIVE_REQUIRED_ENGINE_2_URL='https://github.com/OldUnreal/UnrealTournamentPatches/releases/tag/v469e'

ARCHIVE_REQUIRED_ENGINE_1_NAME='OldUnreal-UTPatch469d-Linux-amd64.tar.bz2'
ARCHIVE_REQUIRED_ENGINE_1_MD5='d0e133165bf1630288583e52a40b90db'
ARCHIVE_REQUIRED_ENGINE_1_SIZE='187239'
ARCHIVE_REQUIRED_ENGINE_1_VERSION='469d-1'
ARCHIVE_REQUIRED_ENGINE_1_URL='https://github.com/OldUnreal/UnrealTournamentPatches/releases/tag/v469d'

ARCHIVE_REQUIRED_ENGINE_0_NAME='OldUnreal-UTPatch469c-Linux-amd64.tar.bz2'
ARCHIVE_REQUIRED_ENGINE_0_MD5='6cd032e70460b1393d9514ffe81dcb1a'
ARCHIVE_REQUIRED_ENGINE_0_VERSION='469c-1'
ARCHIVE_REQUIRED_ENGINE_0_URL='https://github.com/OldUnreal/UnrealTournamentPatches/releases/tag/v469c'

## Unreal Tournament (game installer)
CONTENT_GAME_DATA_PATH='app'
CONTENT_GAME_DATA_FILES='
Maps
Music
Sounds
Textures'
CONTENT_DOC_DATA_PATH='app'
CONTENT_DOC_DATA_FILES='
Help
Manual'

## Native Linux engine
CONTENT_PATH_DEFAULT='.'
CONTENT_LIBS_BIN_RELATIVE_PATH='System64'
CONTENT_LIBS_BIN_FILES='
ALAudio.so
Cluster.so
Core.so
Editor.so
Engine.so
Fire.so
IpDrv.so
libfmod.so.13
OpenGLDrv.so
Render.so
SDLDrv.so
udemo.so
UWeb.so
VulkanDrv.so
XOpenGLDrv.so'
CONTENT_GAME_BIN_FILES='
System
System64/Default.ini
System64/DefUser.ini
System64/Packages.md5
System64/ucc-bin
System64/ucc-bin-amd64
System64/ut-bin
System64/ut-bin-amd64'
## FIXME: Install fonts in the correct path.
CONTENT_GAME0_BIN_FILES='
System64/Fonts/CourierPrime.ttf
System64/Fonts/OpenSans-Regular.ttf
System64/Fonts/Tinos-Regular.ttf'
CONTENT_GAME0_DATA_FILES='
SystemLocalized
Textures
Web'
CONTENT_DOC0_DATA_FILES='
Help
LICENSE.md'

APP_MAIN_EXE='System64/ut-bin-amd64'
APP_MAIN_ICON='app/System/Unreal.ico'

PACKAGES_LIST='
PKG_BIN
PKG_DATA'

PKG_BIN_ARCH='64'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_DATA'
PKG_BIN_DEPENDENCIES_LIBRARIES='
libc.so.6
libdl.so.2
libgcc_s.so.1
libmpg123.so.0
libm.so.6
libopenal.so.1
libpthread.so.0
librt.so.1
libSDL2-2.0.so.0
libsndfile.so.1
libstdc++.so.6
libxmp.so.4'

PKG_DATA_ID="${GAME_ID}-data"
PKG_DATA_DESCRIPTION='data'

# Load common functions.

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run default initialisation actions.

initialization_default "$@"

# Chek for the presence of the game engine archive.

archive_initialize_required \
	'ARCHIVE_ENGINE' \
	'ARCHIVE_REQUIRED_ENGINE_2' \
	'ARCHIVE_REQUIRED_ENGINE_1' \
	'ARCHIVE_REQUIRED_ENGINE_0'
archive_version=$(archive_version 'ARCHIVE_ENGINE' | cut --delimiter='-' --fields=1)
archive_release=$(archive_version "$(current_archive)" | cut --delimiter='-' --fields=2)
export "$(current_archive)_VERSION=${archive_version}-${archive_release}"

# Extract game data

archive_extraction_default
archive_extraction 'ARCHIVE_ENGINE'

# Include game data.

content_inclusion_icons 'PKG_DATA'
content_inclusion_default

# The game engine looks for libmpg123.so instead of libmpg123.so.0.

case "$(option_value 'package')" in
	('arch')
		# Arch Linux already provides the unversioned .so.
	;;
	('gentoo'|'egentoo')
		# Gentoo already provides the unversioned .so.
	;;
	('deb')
		library_destination="$(package_path 'PKG_BIN')$(path_libraries)/libmpg123.so"
		mkdir --parents "$(dirname "$library_destination")"
		ln --symbolic "$(path_libraries_system)/libmpg123.so.0" "$library_destination"
	;;
esac

# Write launchers

## Trying to run the game with wayland SDL backend leads to a crash on launch:
## Critical: Inconsistent SDL window flags
## (SDL 3.2.26)
APP_MAIN_PRERUN='
# Prevent a crash on launch when wayland SDL backend is used.
if [ "${SDL_VIDEODRIVER:-}" = "wayland" ]; then
	unset SDL_VIDEODRIVER
fi
'
## Run the game binary from its parent directory.
game_exec_line() {
	cat <<- 'EOF'
	cd System64
	./ut-bin-amd64 "$@"
	EOF
}

launchers_generation

# Build packages.

packages_generation
print_instructions

# Clean up.

working_directory_cleanup

exit 0
