/************************************************************************
 **
 **  @file   svgdef.cpp
 **  @author Roman Telezhynskyi <dismine(at)gmail.com>
 **  @date   13 6, 2023
 **
 **  @brief
 **  @copyright
 **  This source code is part of the Valentina project, a pattern making
 **  program, whose allow create and modeling patterns of clothing.
 **  Copyright (C) 2023 Valentina project
 **  <https://gitlab.com/smart-pattern/valentina> All Rights Reserved.
 **
 **  Valentina is free software: you can redistribute it and/or modify
 **  it under the terms of the GNU General Public License as published by
 **  the Free Software Foundation, either version 3 of the License, or
 **  (at your option) any later version.
 **
 **  Valentina is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **
 **  You should have received a copy of the GNU General Public License
 **  along with Valentina.  If not, see <http://www.gnu.org/licenses/>.
 **
 *************************************************************************/

#include "svgdef.h"

#include <array>
#include <QChar>
#include <QString>

#if QT_VERSION < QT_VERSION_CHECK(6, 4, 0)
#include "../compatibility.h"
#endif

using namespace Qt::Literals::StringLiterals;

namespace
{
struct Range
{
    constexpr Range(char16_t s, char16_t e)
      : start(s),
        end(e)
    {
    }

    char16_t start;
    char16_t end;
};

constexpr std::array<Range, 1749> simplifiedChineseWritingSystemRanges = {{
    {0x0020, 0x005F}, {0x007B, 0x007D}, {0x00A0, 0x00A0}, {0x00A7, 0x00A7}, {0x00A9, 0x00A9}, {0x00B7, 0x00B7},
    {0x2010, 0x2011}, {0x2018, 0x2019}, {0x2025, 0x2026}, {0x2030, 0x2030}, {0x2032, 0x2033}, {0x2035, 0x2035},
    {0x203B, 0x203B}, {0x3001, 0x3003}, {0x3007, 0x3011}, {0x3014, 0x3017}, {0x301D, 0x301E}, {0x4E00, 0x4E01},
    {0x4E03, 0x4E03}, {0x4E07, 0x4E0E}, {0x4E11, 0x4E11}, {0x4E13, 0x4E14}, {0x4E16, 0x4E16}, {0x4E18, 0x4E1A},
    {0x4E1C, 0x4E1D}, {0x4E22, 0x4E22}, {0x4E24, 0x4E25}, {0x4E27, 0x4E27}, {0x4E2A, 0x4E2A}, {0x4E2D, 0x4E2D},
    {0x4E30, 0x4E30}, {0x4E32, 0x4E32}, {0x4E34, 0x4E34}, {0x4E38, 0x4E3B}, {0x4E3D, 0x4E3E}, {0x4E43, 0x4E43},
    {0x4E45, 0x4E45}, {0x4E48, 0x4E49}, {0x4E4B, 0x4E50}, {0x4E54, 0x4E54}, {0x4E56, 0x4E56}, {0x4E58, 0x4E59},
    {0x4E5D, 0x4E5D}, {0x4E5F, 0x4E61}, {0x4E66, 0x4E66}, {0x4E70, 0x4E71}, {0x4E7E, 0x4E7E}, {0x4E86, 0x4E86},
    {0x4E88, 0x4E89}, {0x4E8B, 0x4E8C}, {0x4E8E, 0x4E8F}, {0x4E91, 0x4E92}, {0x4E94, 0x4E95}, {0x4E9A, 0x4E9B},
    {0x4EA1, 0x4EA1}, {0x4EA4, 0x4EA8}, {0x4EAB, 0x4EAC}, {0x4EAE, 0x4EAE}, {0x4EB2, 0x4EB2}, {0x4EBA, 0x4EBA},
    {0x4EBF, 0x4EC1}, {0x4EC5, 0x4EC5}, {0x4EC7, 0x4EC7}, {0x4ECA, 0x4ECB}, {0x4ECD, 0x4ECE}, {0x4ED4, 0x4ED4},
    {0x4ED6, 0x4ED6}, {0x4ED8, 0x4ED9}, {0x4EE3, 0x4EE5}, {0x4EEA, 0x4EEA}, {0x4EEC, 0x4EEC}, {0x4EF0, 0x4EF0},
    {0x4EF2, 0x4EF2}, {0x4EF6, 0x4EF7}, {0x4EFB, 0x4EFB}, {0x4EFD, 0x4EFD}, {0x4EFF, 0x4EFF}, {0x4F01, 0x4F01},
    {0x4F0A, 0x4F0A}, {0x4F0D, 0x4F0D}, {0x4F0F, 0x4F11}, {0x4F17, 0x4F1A}, {0x4F1F, 0x4F20}, {0x4F24, 0x4F24},
    {0x4F26, 0x4F26}, {0x4F2F, 0x4F30}, {0x4F34, 0x4F34}, {0x4F38, 0x4F38}, {0x4F3C, 0x4F3D}, {0x4F46, 0x4F46},
    {0x4F4D, 0x4F51}, {0x4F53, 0x4F53}, {0x4F55, 0x4F55}, {0x4F59, 0x4F59}, {0x4F5B, 0x4F5C}, {0x4F60, 0x4F60},
    {0x4F64, 0x4F64}, {0x4F69, 0x4F69}, {0x4F73, 0x4F73}, {0x4F7F, 0x4F7F}, {0x4F8B, 0x4F8B}, {0x4F9B, 0x4F9B},
    {0x4F9D, 0x4F9D}, {0x4FA0, 0x4FA0}, {0x4FA6, 0x4FA8}, {0x4FAC, 0x4FAC}, {0x4FAF, 0x4FAF}, {0x4FB5, 0x4FB5},
    {0x4FBF, 0x4FBF}, {0x4FC3, 0x4FC4}, {0x4FCA, 0x4FCA}, {0x4FD7, 0x4FD7}, {0x4FDD, 0x4FDD}, {0x4FE1, 0x4FE1},
    {0x4FE9, 0x4FE9}, {0x4FEE, 0x4FEE}, {0x4FF1, 0x4FF1}, {0x4FFE, 0x4FFE}, {0x500D, 0x500D}, {0x5012, 0x5012},
    {0x5019, 0x501A}, {0x501F, 0x501F}, {0x5026, 0x5026}, {0x503C, 0x503C}, {0x503E, 0x503E}, {0x5047, 0x5047},
    {0x504C, 0x504C}, {0x504F, 0x504F}, {0x505A, 0x505A}, {0x505C, 0x505C}, {0x5065, 0x5065}, {0x5076, 0x5077},
    {0x50A8, 0x50A8}, {0x50AC, 0x50AC}, {0x50B2, 0x50B2}, {0x50BB, 0x50BB}, {0x50CF, 0x50CF}, {0x50E7, 0x50E7},
    {0x5112, 0x5112}, {0x513F, 0x513F}, {0x5141, 0x5141}, {0x5143, 0x5146}, {0x5148, 0x5149}, {0x514B, 0x514B},
    {0x514D, 0x514D}, {0x5151, 0x5151}, {0x5154, 0x5154}, {0x515A, 0x515A}, {0x5165, 0x5165}, {0x5168, 0x5168},
    {0x516B, 0x516E}, {0x5170, 0x5171}, {0x5173, 0x5179}, {0x517B, 0x517D}, {0x5185, 0x5185}, {0x5188, 0x5188},
    {0x518C, 0x518D}, {0x5192, 0x5192}, {0x5199, 0x5199}, {0x519B, 0x519C}, {0x51A0, 0x51A0}, {0x51AC, 0x51AC},
    {0x51B0, 0x51B0}, {0x51B2, 0x51B3}, {0x51B5, 0x51B5}, {0x51B7, 0x51B7}, {0x51C6, 0x51C6}, {0x51CC, 0x51CC},
    {0x51CF, 0x51CF}, {0x51DD, 0x51DD}, {0x51E0, 0x51E1}, {0x51E4, 0x51E4}, {0x51ED, 0x51ED}, {0x51EF, 0x51F0},
    {0x51FA, 0x51FB}, {0x51FD, 0x51FD}, {0x5200, 0x5200}, {0x5206, 0x5207}, {0x520A, 0x520A}, {0x5211, 0x5212},
    {0x5217, 0x521B}, {0x521D, 0x521D}, {0x5224, 0x5224}, {0x5229, 0x5229}, {0x522B, 0x522B}, {0x5230, 0x5230},
    {0x5236, 0x5238}, {0x523A, 0x523B}, {0x5242, 0x5242}, {0x524D, 0x524D}, {0x5251, 0x5251}, {0x5267, 0x5267},
    {0x5269, 0x526A}, {0x526F, 0x526F}, {0x5272, 0x5272}, {0x529B, 0x529B}, {0x529D, 0x52A1}, {0x52A3, 0x52A3},
    {0x52A8, 0x52AB}, {0x52B1, 0x52B3}, {0x52BF, 0x52BF}, {0x52C7, 0x52C7}, {0x52C9, 0x52C9}, {0x52CB, 0x52CB},
    {0x52D2, 0x52D2}, {0x52E4, 0x52E4}, {0x52FE, 0x52FF}, {0x5305, 0x5306}, {0x5308, 0x5308}, {0x5316, 0x5317},
    {0x5319, 0x5319}, {0x5339, 0x533B}, {0x5341, 0x5341}, {0x5343, 0x5343}, {0x5347, 0x5348}, {0x534A, 0x534A},
    {0x534E, 0x534F}, {0x5352, 0x5353}, {0x5355, 0x5357}, {0x535A, 0x535A}, {0x5360, 0x5362}, {0x536B, 0x536B},
    {0x536F, 0x5371}, {0x5373, 0x5374}, {0x5377, 0x5377}, {0x5382, 0x5382}, {0x5384, 0x5386}, {0x5389, 0x5389},
    {0x538B, 0x538D}, {0x539A, 0x539A}, {0x539F, 0x539F}, {0x53BB, 0x53BB}, {0x53BF, 0x53BF}, {0x53C2, 0x53C2},
    {0x53C8, 0x53CD}, {0x53D1, 0x53D1}, {0x53D4, 0x53D4}, {0x53D6, 0x53D9}, {0x53E3, 0x53E6}, {0x53EA, 0x53ED},
    {0x53EF, 0x53F0}, {0x53F2, 0x53F3}, {0x53F6, 0x53F9}, {0x5403, 0x5404}, {0x5408, 0x540A}, {0x540C, 0x540E},
    {0x5410, 0x5411}, {0x5413, 0x5413}, {0x5417, 0x5417}, {0x541B, 0x541B}, {0x541D, 0x541D}, {0x541F, 0x541F},
    {0x5426, 0x5427}, {0x542B, 0x542C}, {0x542F, 0x542F}, {0x5435, 0x5435}, {0x5438, 0x5439}, {0x543B, 0x543B},
    {0x543E, 0x543E}, {0x5440, 0x5440}, {0x5446, 0x5446}, {0x5448, 0x5448}, {0x544A, 0x544A}, {0x5450, 0x5450},
    {0x5458, 0x5458}, {0x545C, 0x545C}, {0x5462, 0x5462}, {0x5466, 0x5466}, {0x5468, 0x5468}, {0x5473, 0x5473},
    {0x5475, 0x5475}, {0x547C, 0x547D}, {0x548C, 0x548C}, {0x5496, 0x5496}, {0x54A6, 0x54A8}, {0x54AA, 0x54AA},
    {0x54AC, 0x54AC}, {0x54AF, 0x54AF}, {0x54B1, 0x54B1}, {0x54C0, 0x54C1}, {0x54C7, 0x54C9}, {0x54CD, 0x54CE},
    {0x54DF, 0x54DF}, {0x54E5, 0x54E6}, {0x54E9, 0x54EA}, {0x54ED, 0x54ED}, {0x54F2, 0x54F2}, {0x5509, 0x5509},
    {0x5510, 0x5510}, {0x5524, 0x5524}, {0x552C, 0x552C}, {0x552E, 0x552F}, {0x5531, 0x5531}, {0x5537, 0x5537},
    {0x5546, 0x5546}, {0x554A, 0x554A}, {0x5561, 0x5561}, {0x5565, 0x5566}, {0x556A, 0x556A}, {0x5580, 0x5580},
    {0x5582, 0x5582}, {0x5584, 0x5584}, {0x5587, 0x5587}, {0x558A, 0x558A}, {0x558F, 0x558F}, {0x5594, 0x5594},
    {0x559C, 0x559D}, {0x55B5, 0x55B5}, {0x55B7, 0x55B7}, {0x55BB, 0x55BB}, {0x55D2, 0x55D2}, {0x55E8, 0x55E8},
    {0x55EF, 0x55EF}, {0x5609, 0x5609}, {0x561B, 0x561B}, {0x5634, 0x5634}, {0x563B, 0x563B}, {0x563F, 0x563F},
    {0x5668, 0x5668}, {0x56DB, 0x56DB}, {0x56DE, 0x56DE}, {0x56E0, 0x56E0}, {0x56E2, 0x56E2}, {0x56ED, 0x56ED},
    {0x56F0, 0x56F0}, {0x56F4, 0x56F4}, {0x56FA, 0x56FA}, {0x56FD, 0x56FE}, {0x5706, 0x5706}, {0x5708, 0x5708},
    {0x571F, 0x571F}, {0x5723, 0x5723}, {0x5728, 0x5728}, {0x572D, 0x572D}, {0x5730, 0x5730}, {0x5733, 0x5733},
    {0x573A, 0x573A}, {0x573E, 0x573E}, {0x5740, 0x5740}, {0x5747, 0x5747}, {0x574E, 0x574E}, {0x5750, 0x5751},
    {0x5757, 0x5757}, {0x575A, 0x575C}, {0x5761, 0x5761}, {0x5764, 0x5764}, {0x5766, 0x5766}, {0x576A, 0x576A},
    {0x5782, 0x5783}, {0x578B, 0x578B}, {0x5792, 0x5792}, {0x57C3, 0x57C3}, {0x57CB, 0x57CB}, {0x57CE, 0x57CE},
    {0x57D4, 0x57D4}, {0x57DF, 0x57DF}, {0x57F9, 0x57FA}, {0x5802, 0x5802}, {0x5806, 0x5806}, {0x5815, 0x5815},
    {0x5821, 0x5821}, {0x582A, 0x582A}, {0x5851, 0x5851}, {0x5854, 0x5854}, {0x585E, 0x585E}, {0x586B, 0x586B},
    {0x5883, 0x5883}, {0x589E, 0x589E}, {0x58A8, 0x58A8}, {0x58C1, 0x58C1}, {0x58E4, 0x58E4}, {0x58EB, 0x58EC},
    {0x58EE, 0x58EE}, {0x58F0, 0x58F0}, {0x5904, 0x5904}, {0x5907, 0x5907}, {0x590D, 0x590D}, {0x590F, 0x590F},
    {0x5915, 0x5916}, {0x591A, 0x591A}, {0x591C, 0x591C}, {0x591F, 0x591F}, {0x5925, 0x5925}, {0x5927, 0x5927},
    {0x5929, 0x592B}, {0x592E, 0x592E}, {0x5931, 0x5931}, {0x5934, 0x5934}, {0x5937, 0x593A}, {0x5947, 0x5949},
    {0x594B, 0x594B}, {0x594F, 0x594F}, {0x5951, 0x5951}, {0x5954, 0x5954}, {0x5956, 0x5957}, {0x5965, 0x5965},
    {0x5973, 0x5974}, {0x5976, 0x5976}, {0x5979, 0x5979}, {0x597D, 0x597D}, {0x5982, 0x5982}, {0x5987, 0x5988},
    {0x5996, 0x5996}, {0x5999, 0x5999}, {0x59A5, 0x59A5}, {0x59A8, 0x59A8}, {0x59AE, 0x59AE}, {0x59B9, 0x59B9},
    {0x59BB, 0x59BB}, {0x59C6, 0x59C6}, {0x59CA, 0x59CB}, {0x59D0, 0x59D1}, {0x59D3, 0x59D4}, {0x59FF, 0x59FF},
    {0x5A01, 0x5A01}, {0x5A03, 0x5A04}, {0x5A18, 0x5A18}, {0x5A1C, 0x5A1C}, {0x5A1F, 0x5A1F}, {0x5A31, 0x5A31},
    {0x5A46, 0x5A46}, {0x5A5A, 0x5A5A}, {0x5A92, 0x5A92}, {0x5AC1, 0x5AC1}, {0x5ACC, 0x5ACC}, {0x5AE9, 0x5AE9},
    {0x5B50, 0x5B50}, {0x5B54, 0x5B55}, {0x5B57, 0x5B59}, {0x5B5C, 0x5B5D}, {0x5B5F, 0x5B5F}, {0x5B63, 0x5B64},
    {0x5B66, 0x5B66}, {0x5B69, 0x5B69}, {0x5B81, 0x5B81}, {0x5B83, 0x5B83}, {0x5B87, 0x5B89}, {0x5B8B, 0x5B8C},
    {0x5B8F, 0x5B8F}, {0x5B97, 0x5B9E}, {0x5BA1, 0x5BA4}, {0x5BAA, 0x5BAA}, {0x5BB3, 0x5BB4}, {0x5BB6, 0x5BB6},
    {0x5BB9, 0x5BB9}, {0x5BBD, 0x5BBF}, {0x5BC2, 0x5BC2}, {0x5BC4, 0x5BC7}, {0x5BCC, 0x5BCC}, {0x5BD2, 0x5BD2},
    {0x5BDD, 0x5BDF}, {0x5BE1, 0x5BE1}, {0x5BE8, 0x5BE8}, {0x5BF8, 0x5BF9}, {0x5BFB, 0x5BFC}, {0x5BFF, 0x5BFF},
    {0x5C01, 0x5C01}, {0x5C04, 0x5C04}, {0x5C06, 0x5C06}, {0x5C0A, 0x5C0A}, {0x5C0F, 0x5C0F}, {0x5C11, 0x5C11},
    {0x5C14, 0x5C14}, {0x5C16, 0x5C16}, {0x5C18, 0x5C18}, {0x5C1A, 0x5C1A}, {0x5C1D, 0x5C1D}, {0x5C24, 0x5C24},
    {0x5C31, 0x5C31}, {0x5C3A, 0x5C3A}, {0x5C3C, 0x5C3E}, {0x5C40, 0x5C42}, {0x5C45, 0x5C45}, {0x5C4B, 0x5C4B},
    {0x5C4F, 0x5C4F}, {0x5C55, 0x5C55}, {0x5C5E, 0x5C5E}, {0x5C60, 0x5C60}, {0x5C71, 0x5C71}, {0x5C81, 0x5C82},
    {0x5C97, 0x5C98}, {0x5C9A, 0x5C9B}, {0x5CB3, 0x5CB3}, {0x5CB8, 0x5CB8}, {0x5CE1, 0x5CE1}, {0x5CF0, 0x5CF0},
    {0x5D07, 0x5D07}, {0x5D29, 0x5D29}, {0x5D34, 0x5D34}, {0x5DDD, 0x5DDE}, {0x5DE1, 0x5DE1}, {0x5DE5, 0x5DE8},
    {0x5DEB, 0x5DEB}, {0x5DEE, 0x5DEE}, {0x5DF1, 0x5DF4}, {0x5DF7, 0x5DF7}, {0x5E01, 0x5E03}, {0x5E05, 0x5E05},
    {0x5E08, 0x5E08}, {0x5E0C, 0x5E0C}, {0x5E10, 0x5E10}, {0x5E15, 0x5E16}, {0x5E1D, 0x5E1D}, {0x5E26, 0x5E26},
    {0x5E2D, 0x5E2E}, {0x5E38, 0x5E38}, {0x5E3D, 0x5E3D}, {0x5E45, 0x5E45}, {0x5E55, 0x5E55}, {0x5E72, 0x5E74},
    {0x5E76, 0x5E76}, {0x5E78, 0x5E78}, {0x5E7B, 0x5E7D}, {0x5E7F, 0x5E7F}, {0x5E86, 0x5E86}, {0x5E8A, 0x5E8A},
    {0x5E8F, 0x5E8F}, {0x5E93, 0x5E95}, {0x5E97, 0x5E97}, {0x5E99, 0x5E9A}, {0x5E9C, 0x5E9C}, {0x5E9E, 0x5E9F},
    {0x5EA6, 0x5EA7}, {0x5EAD, 0x5EAD}, {0x5EB7, 0x5EB8}, {0x5EC9, 0x5EC9}, {0x5ED6, 0x5ED6}, {0x5EF6, 0x5EF7},
    {0x5EFA, 0x5EFA}, {0x5F00, 0x5F00}, {0x5F02, 0x5F04}, {0x5F0A, 0x5F0A}, {0x5F0F, 0x5F0F}, {0x5F15, 0x5F15},
    {0x5F17, 0x5F18}, {0x5F1F, 0x5F20}, {0x5F25, 0x5F26}, {0x5F2F, 0x5F2F}, {0x5F31, 0x5F31}, {0x5F39, 0x5F3A},
    {0x5F52, 0x5F53}, {0x5F55, 0x5F55}, {0x5F5D, 0x5F5D}, {0x5F62, 0x5F62}, {0x5F69, 0x5F69}, {0x5F6C, 0x5F6D},
    {0x5F70, 0x5F71}, {0x5F77, 0x5F77}, {0x5F79, 0x5F79}, {0x5F7B, 0x5F7C}, {0x5F80, 0x5F81}, {0x5F84, 0x5F85},
    {0x5F88, 0x5F88}, {0x5F8B, 0x5F8C}, {0x5F90, 0x5F90}, {0x5F92, 0x5F92}, {0x5F97, 0x5F97}, {0x5FAA, 0x5FAA},
    {0x5FAE, 0x5FAE}, {0x5FB5, 0x5FB5}, {0x5FB7, 0x5FB7}, {0x5FC3, 0x5FC3}, {0x5FC5, 0x5FC6}, {0x5FCC, 0x5FCD},
    {0x5FD7, 0x5FD9}, {0x5FE0, 0x5FE0}, {0x5FE7, 0x5FE7}, {0x5FEB, 0x5FEB}, {0x5FF5, 0x5FF5}, {0x5FFD, 0x5FFD},
    {0x6000, 0x6001}, {0x600E, 0x600E}, {0x6012, 0x6012}, {0x6015, 0x6016}, {0x601D, 0x601D}, {0x6021, 0x6021},
    {0x6025, 0x6025}, {0x6027, 0x6028}, {0x602A, 0x602A}, {0x603B, 0x603B}, {0x604B, 0x604B}, {0x6050, 0x6050},
    {0x6062, 0x6062}, {0x6068, 0x6069}, {0x606D, 0x606D}, {0x606F, 0x6070}, {0x6076, 0x6076}, {0x607C, 0x607C},
    {0x6084, 0x6084}, {0x6089, 0x6089}, {0x6094, 0x6094}, {0x609F, 0x60A0}, {0x60A3, 0x60A3}, {0x60A8, 0x60A8},
    {0x60B2, 0x60B2}, {0x60C5, 0x60C5}, {0x60D1, 0x60D1}, {0x60DC, 0x60DC}, {0x60E0, 0x60E0}, {0x60E7, 0x60E8},
    {0x60EF, 0x60EF}, {0x60F3, 0x60F3}, {0x60F9, 0x60F9}, {0x6101, 0x6101}, {0x6108, 0x6109}, {0x610F, 0x610F},
    {0x611A, 0x611A}, {0x611F, 0x611F}, {0x6127, 0x6127}, {0x6148, 0x6148}, {0x614E, 0x614E}, {0x6155, 0x6155},
    {0x6162, 0x6162}, {0x6167, 0x6167}, {0x6170, 0x6170}, {0x61BE, 0x61BE}, {0x61C2, 0x61C2}, {0x61D2, 0x61D2},
    {0x6208, 0x6208}, {0x620A, 0x620A}, {0x620C, 0x620C}, {0x620F, 0x6212}, {0x6216, 0x6216}, {0x6218, 0x6218},
    {0x622A, 0x622A}, {0x6234, 0x6234}, {0x6237, 0x6237}, {0x623F, 0x6241}, {0x6247, 0x6247}, {0x624B, 0x624B},
    {0x624D, 0x624E}, {0x6251, 0x6251}, {0x6253, 0x6253}, {0x6258, 0x6258}, {0x6263, 0x6263}, {0x6267, 0x6267},
    {0x6269, 0x6269}, {0x626B, 0x626F}, {0x6279, 0x6279}, {0x627E, 0x6280}, {0x6284, 0x6284}, {0x628A, 0x628A},
    {0x6291, 0x6291}, {0x6293, 0x6293}, {0x6295, 0x6295}, {0x6297, 0x6298}, {0x62A2, 0x62A2}, {0x62A4, 0x62A5},
    {0x62AB, 0x62AC}, {0x62B1, 0x62B1}, {0x62B5, 0x62B5}, {0x62B9, 0x62B9}, {0x62BD, 0x62BD}, {0x62C5, 0x62C6},
    {0x62C9, 0x62C9}, {0x62CD, 0x62CD}, {0x62D2, 0x62D2}, {0x62D4, 0x62D4}, {0x62D6, 0x62D6}, {0x62D8, 0x62D8},
    {0x62DB, 0x62DC}, {0x62DF, 0x62DF}, {0x62E5, 0x62E6}, {0x62E8, 0x62E9}, {0x62EC, 0x62EC}, {0x62F3, 0x62F3},
    {0x62F7, 0x62F7}, {0x62FC, 0x62FC}, {0x62FE, 0x62FF}, {0x6301, 0x6301}, {0x6307, 0x6307}, {0x6309, 0x6309},
    {0x6311, 0x6311}, {0x6316, 0x6316}, {0x631D, 0x631D}, {0x6321, 0x6321}, {0x6324, 0x6325}, {0x632A, 0x632A},
    {0x632F, 0x632F}, {0x633A, 0x633A}, {0x6349, 0x6349}, {0x6350, 0x6350}, {0x6355, 0x6355}, {0x635F, 0x635F},
    {0x6361, 0x6362}, {0x636E, 0x636E}, {0x6377, 0x6377}, {0x6388, 0x6389}, {0x638C, 0x638C}, {0x6392, 0x6392},
    {0x63A2, 0x63A2}, {0x63A5, 0x63A5}, {0x63A7, 0x63AA}, {0x63B8, 0x63B8}, {0x63CF, 0x63D0}, {0x63D2, 0x63D2},
    {0x63E1, 0x63E1}, {0x63F4, 0x63F4}, {0x641C, 0x641C}, {0x641E, 0x641E}, {0x642C, 0x642D}, {0x6444, 0x6444},
    {0x6446, 0x6446}, {0x644A, 0x644A}, {0x6454, 0x6454}, {0x6458, 0x6458}, {0x6469, 0x6469}, {0x6478, 0x6478},
    {0x6492, 0x6492}, {0x649E, 0x649E}, {0x64AD, 0x64AD}, {0x64CD, 0x64CE}, {0x64E6, 0x64E6}, {0x652F, 0x652F},
    {0x6536, 0x6536}, {0x6539, 0x6539}, {0x653B, 0x653B}, {0x653E, 0x653F}, {0x6545, 0x6545}, {0x6548, 0x6548},
    {0x654C, 0x654C}, {0x654F, 0x654F}, {0x6551, 0x6551}, {0x6559, 0x6559}, {0x655D, 0x655D}, {0x6562, 0x6563},
    {0x6566, 0x6566}, {0x656C, 0x656C}, {0x6570, 0x6570}, {0x6572, 0x6572}, {0x6574, 0x6574}, {0x6587, 0x6587},
    {0x658B, 0x658B}, {0x6590, 0x6590}, {0x6597, 0x6597}, {0x6599, 0x6599}, {0x659C, 0x659C}, {0x65A5, 0x65A5},
    {0x65AD, 0x65AD}, {0x65AF, 0x65B0}, {0x65B9, 0x65B9}, {0x65BC, 0x65BD}, {0x65C1, 0x65C1}, {0x65C5, 0x65C5},
    {0x65CB, 0x65CB}, {0x65CF, 0x65CF}, {0x65D7, 0x65D7}, {0x65E0, 0x65E0}, {0x65E2, 0x65E2}, {0x65E5, 0x65E9},
    {0x65ED, 0x65ED}, {0x65F6, 0x65F6}, {0x65FA, 0x65FA}, {0x6602, 0x6602}, {0x6606, 0x6606}, {0x660C, 0x660C},
    {0x660E, 0x660F}, {0x6613, 0x6613}, {0x661F, 0x6620}, {0x6625, 0x6625}, {0x6628, 0x6628}, {0x662D, 0x662D},
    {0x662F, 0x662F}, {0x663E, 0x663E}, {0x6643, 0x6643}, {0x664B, 0x664B}, {0x6652, 0x6653}, {0x665A, 0x665A},
    {0x6668, 0x6668}, {0x666E, 0x666F}, {0x6674, 0x6674}, {0x6676, 0x6676}, {0x667A, 0x667A}, {0x6682, 0x6682},
    {0x6691, 0x6691}, {0x6696, 0x6697}, {0x66AE, 0x66AE}, {0x66B4, 0x66B4}, {0x66F0, 0x66F0}, {0x66F2, 0x66F2},
    {0x66F4, 0x66F4}, {0x66F9, 0x66F9}, {0x66FC, 0x66FC}, {0x66FE, 0x6700}, {0x6708, 0x6709}, {0x670B, 0x670B},
    {0x670D, 0x670D}, {0x6717, 0x6717}, {0x671B, 0x671B}, {0x671D, 0x671D}, {0x671F, 0x671F}, {0x6728, 0x6728},
    {0x672A, 0x672D}, {0x672F, 0x672F}, {0x6731, 0x6731}, {0x6735, 0x6735}, {0x673A, 0x673A}, {0x6740, 0x6740},
    {0x6742, 0x6743}, {0x6749, 0x6749}, {0x674E, 0x674E}, {0x6750, 0x6751}, {0x675C, 0x675C}, {0x675F, 0x675F},
    {0x6761, 0x6761}, {0x6765, 0x6765}, {0x6768, 0x6768}, {0x676F, 0x6770}, {0x677E, 0x677F}, {0x6781, 0x6781},
    {0x6784, 0x6784}, {0x6790, 0x6790}, {0x6797, 0x6797}, {0x679C, 0x679D}, {0x67A2, 0x67A2}, {0x67AA, 0x67AB},
    {0x67B6, 0x67B6}, {0x67CF, 0x67D0}, {0x67D3, 0x67D4}, {0x67E5, 0x67E5}, {0x67EC, 0x67EC}, {0x67EF, 0x67EF},
    {0x67F3, 0x67F4}, {0x6807, 0x6807}, {0x680B, 0x680B}, {0x680F, 0x680F}, {0x6811, 0x6811}, {0x6821, 0x6821},
    {0x6837, 0x6839}, {0x683C, 0x683C}, {0x6843, 0x6843}, {0x6846, 0x6846}, {0x6848, 0x6848}, {0x684C, 0x684C},
    {0x6851, 0x6851}, {0x6863, 0x6863}, {0x6865, 0x6865}, {0x6881, 0x6881}, {0x6885, 0x6885}, {0x68A6, 0x68A6},
    {0x68AF, 0x68B0}, {0x68B5, 0x68B5}, {0x68C0, 0x68C0}, {0x68C9, 0x68C9}, {0x68CB, 0x68CB}, {0x68D2, 0x68D2},
    {0x68DA, 0x68DA}, {0x68EE, 0x68EE}, {0x6905, 0x6905}, {0x690D, 0x690D}, {0x6930, 0x6930}, {0x695A, 0x695A},
    {0x697C, 0x697C}, {0x6982, 0x6982}, {0x699C, 0x699C}, {0x6A21, 0x6A21}, {0x6A31, 0x6A31}, {0x6A80, 0x6A80},
    {0x6B20, 0x6B23}, {0x6B27, 0x6B27}, {0x6B32, 0x6B32}, {0x6B3A, 0x6B3A}, {0x6B3E, 0x6B3E}, {0x6B49, 0x6B49},
    {0x6B4C, 0x6B4C}, {0x6B62, 0x6B66}, {0x6B6A, 0x6B6A}, {0x6B7B, 0x6B7B}, {0x6B8A, 0x6B8B}, {0x6BB5, 0x6BB5},
    {0x6BC5, 0x6BC5}, {0x6BCD, 0x6BCD}, {0x6BCF, 0x6BCF}, {0x6BD2, 0x6BD2}, {0x6BD4, 0x6BD5}, {0x6BDB, 0x6BDB},
    {0x6BEB, 0x6BEB}, {0x6C0F, 0x6C0F}, {0x6C11, 0x6C11}, {0x6C14, 0x6C14}, {0x6C1B, 0x6C1B}, {0x6C34, 0x6C34},
    {0x6C38, 0x6C38}, {0x6C42, 0x6C42}, {0x6C47, 0x6C47}, {0x6C49, 0x6C49}, {0x6C57, 0x6C57}, {0x6C5D, 0x6C5D},
    {0x6C5F, 0x6C61}, {0x6C64, 0x6C64}, {0x6C6A, 0x6C6A}, {0x6C76, 0x6C76}, {0x6C7D, 0x6C7D}, {0x6C83, 0x6C83},
    {0x6C88, 0x6C89}, {0x6C99, 0x6C99}, {0x6C9F, 0x6C9F}, {0x6CA1, 0x6CA1}, {0x6CA7, 0x6CA7}, {0x6CB3, 0x6CB3},
    {0x6CB9, 0x6CB9}, {0x6CBB, 0x6CBB}, {0x6CBF, 0x6CBF}, {0x6CC9, 0x6CCA}, {0x6CD5, 0x6CD5}, {0x6CDB, 0x6CDB},
    {0x6CE1, 0x6CE3}, {0x6CE5, 0x6CE5}, {0x6CE8, 0x6CE8}, {0x6CF0, 0x6CF0}, {0x6CF3, 0x6CF3}, {0x6CFD, 0x6CFD},
    {0x6D0B, 0x6D0B}, {0x6D17, 0x6D17}, {0x6D1B, 0x6D1B}, {0x6D1E, 0x6D1E}, {0x6D25, 0x6D25}, {0x6D2A, 0x6D2A},
    {0x6D32, 0x6D32}, {0x6D3B, 0x6D3B}, {0x6D3D, 0x6D3E}, {0x6D41, 0x6D41}, {0x6D45, 0x6D45}, {0x6D4B, 0x6D4B},
    {0x6D4E, 0x6D4F}, {0x6D51, 0x6D51}, {0x6D53, 0x6D53}, {0x6D59, 0x6D59}, {0x6D66, 0x6D66}, {0x6D69, 0x6D6A},
    {0x6D6E, 0x6D6E}, {0x6D74, 0x6D74}, {0x6D77, 0x6D77}, {0x6D85, 0x6D85}, {0x6D88, 0x6D89}, {0x6D9B, 0x6D9B},
    {0x6DA8, 0x6DA8}, {0x6DAF, 0x6DAF}, {0x6DB2, 0x6DB2}, {0x6DB5, 0x6DB5}, {0x6DCB, 0x6DCB}, {0x6DD1, 0x6DD1},
    {0x6DD8, 0x6DD8}, {0x6DE1, 0x6DE1}, {0x6DF1, 0x6DF1}, {0x6DF7, 0x6DF7}, {0x6DFB, 0x6DFB}, {0x6E05, 0x6E05},
    {0x6E10, 0x6E10}, {0x6E21, 0x6E21}, {0x6E23, 0x6E23}, {0x6E29, 0x6E29}, {0x6E2F, 0x6E2F}, {0x6E34, 0x6E34},
    {0x6E38, 0x6E38}, {0x6E56, 0x6E56}, {0x6E7E, 0x6E7E}, {0x6E90, 0x6E90}, {0x6E9C, 0x6E9C}, {0x6EAA, 0x6EAA},
    {0x6ECB, 0x6ECB}, {0x6ED1, 0x6ED1}, {0x6EE1, 0x6EE1}, {0x6EE5, 0x6EE5}, {0x6EE8, 0x6EE8}, {0x6EF4, 0x6EF4},
    {0x6F02, 0x6F02}, {0x6F0F, 0x6F0F}, {0x6F14, 0x6F14}, {0x6F20, 0x6F20}, {0x6F2B, 0x6F2B}, {0x6F58, 0x6F58},
    {0x6F5C, 0x6F5C}, {0x6F6E, 0x6F6E}, {0x6F8E, 0x6F8E}, {0x6FB3, 0x6FB3}, {0x6FC0, 0x6FC0}, {0x704C, 0x704C},
    {0x706B, 0x706B}, {0x706D, 0x706D}, {0x706F, 0x7070}, {0x7075, 0x7075}, {0x707F, 0x707F}, {0x7089, 0x7089},
    {0x708E, 0x708E}, {0x70AE, 0x70AE}, {0x70B8, 0x70B9}, {0x70C2, 0x70C2}, {0x70C8, 0x70C8}, {0x70E4, 0x70E4},
    {0x70E6, 0x70E7}, {0x70ED, 0x70ED}, {0x7126, 0x7126}, {0x7136, 0x7136}, {0x714C, 0x714C}, {0x715E, 0x715E},
    {0x7167, 0x7167}, {0x716E, 0x716E}, {0x718A, 0x718A}, {0x719F, 0x719F}, {0x71C3, 0x71C3}, {0x71D5, 0x71D5},
    {0x7206, 0x7206}, {0x722A, 0x722A}, {0x722C, 0x722C}, {0x7231, 0x7231}, {0x7235, 0x7238}, {0x723D, 0x723D},
    {0x7247, 0x7248}, {0x724C, 0x724C}, {0x7259, 0x7259}, {0x725B, 0x725B}, {0x7261, 0x7262}, {0x7267, 0x7267},
    {0x7269, 0x7269}, {0x7272, 0x7272}, {0x7275, 0x7275}, {0x7279, 0x727A}, {0x72AF, 0x72AF}, {0x72B6, 0x72B6},
    {0x72B9, 0x72B9}, {0x72C2, 0x72C2}, {0x72D0, 0x72D0}, {0x72D7, 0x72D7}, {0x72E0, 0x72E0}, {0x72EC, 0x72EC},
    {0x72EE, 0x72EE}, {0x72F1, 0x72F1}, {0x72FC, 0x72FC}, {0x731B, 0x731C}, {0x732A, 0x732A}, {0x732E, 0x732E},
    {0x7334, 0x7334}, {0x7384, 0x7384}, {0x7387, 0x7387}, {0x7389, 0x7389}, {0x738B, 0x738B}, {0x739B, 0x739B},
    {0x73A9, 0x73A9}, {0x73AB, 0x73AB}, {0x73AF, 0x73B0}, {0x73B2, 0x73B2}, {0x73BB, 0x73BB}, {0x73C0, 0x73C0},
    {0x73CA, 0x73CA}, {0x73CD, 0x73CD}, {0x73E0, 0x73E0}, {0x73ED, 0x73ED}, {0x7403, 0x7403}, {0x7406, 0x7406},
    {0x740A, 0x740A}, {0x742A, 0x742A}, {0x7433, 0x7434}, {0x743C, 0x743C}, {0x7459, 0x7459}, {0x745C, 0x745C},
    {0x745E, 0x745F}, {0x7470, 0x7470}, {0x7476, 0x7476}, {0x7483, 0x7483}, {0x74DC, 0x74DC}, {0x74E6, 0x74E6},
    {0x74F6, 0x74F6}, {0x7518, 0x7518}, {0x751A, 0x751A}, {0x751C, 0x751C}, {0x751F, 0x751F}, {0x7528, 0x7528},
    {0x7530, 0x7533}, {0x7535, 0x7535}, {0x7537, 0x7538}, {0x753B, 0x753B}, {0x7545, 0x7545}, {0x754C, 0x754C},
    {0x7559, 0x7559}, {0x7565, 0x7565}, {0x756A, 0x756A}, {0x7586, 0x7586}, {0x758F, 0x758F}, {0x7591, 0x7591},
    {0x7597, 0x7597}, {0x75AF, 0x75AF}, {0x75B2, 0x75B2}, {0x75BC, 0x75BC}, {0x75BE, 0x75BE}, {0x75C5, 0x75C5},
    {0x75D5, 0x75D5}, {0x75DB, 0x75DB}, {0x75F4, 0x75F4}, {0x7678, 0x7678}, {0x767B, 0x767B}, {0x767D, 0x767E},
    {0x7684, 0x7684}, {0x7686, 0x7687}, {0x76AE, 0x76AE}, {0x76C8, 0x76C8}, {0x76CA, 0x76CA}, {0x76D1, 0x76D2},
    {0x76D6, 0x76D6}, {0x76D8, 0x76D8}, {0x76DB, 0x76DB}, {0x76DF, 0x76DF}, {0x76EE, 0x76EE}, {0x76F4, 0x76F4},
    {0x76F8, 0x76F8}, {0x76FC, 0x76FC}, {0x76FE, 0x76FE}, {0x7701, 0x7701}, {0x7709, 0x7709}, {0x770B, 0x770B},
    {0x771F, 0x7720}, {0x773C, 0x773C}, {0x7740, 0x7740}, {0x775B, 0x775B}, {0x7761, 0x7761}, {0x7763, 0x7763},
    {0x77A7, 0x77A7}, {0x77DB, 0x77DB}, {0x77E3, 0x77E3}, {0x77E5, 0x77E5}, {0x77ED, 0x77ED}, {0x77F3, 0x77F3},
    {0x77F6, 0x77F6}, {0x7801, 0x7802}, {0x780D, 0x780D}, {0x7814, 0x7814}, {0x7834, 0x7834}, {0x7840, 0x7840},
    {0x7855, 0x7855}, {0x786C, 0x786C}, {0x786E, 0x786E}, {0x788D, 0x788E}, {0x7897, 0x7897}, {0x789F, 0x789F},
    {0x78A7, 0x78A7}, {0x78B0, 0x78B0}, {0x78C1, 0x78C1}, {0x78C5, 0x78C5}, {0x78E8, 0x78E8}, {0x793A, 0x793A},
    {0x793C, 0x793C}, {0x793E, 0x793E}, {0x7956, 0x7956}, {0x795A, 0x795A}, {0x795D, 0x795E}, {0x7965, 0x7965},
    {0x7968, 0x7968}, {0x796F, 0x796F}, {0x7978, 0x7978}, {0x7981, 0x7981}, {0x7985, 0x7985}, {0x798F, 0x798F},
    {0x79BB, 0x79BB}, {0x79C0, 0x79C1}, {0x79CB, 0x79CB}, {0x79CD, 0x79CD}, {0x79D1, 0x79D2}, {0x79D8, 0x79D8},
    {0x79DF, 0x79DF}, {0x79E4, 0x79E4}, {0x79E6, 0x79E6}, {0x79E9, 0x79E9}, {0x79EF, 0x79F0}, {0x79FB, 0x79FB},
    {0x7A00, 0x7A00}, {0x7A0B, 0x7A0B}, {0x7A0D, 0x7A0E}, {0x7A23, 0x7A23}, {0x7A33, 0x7A33}, {0x7A3F, 0x7A3F},
    {0x7A46, 0x7A46}, {0x7A76, 0x7A77}, {0x7A79, 0x7A7A}, {0x7A7F, 0x7A7F}, {0x7A81, 0x7A81}, {0x7A97, 0x7A97},
    {0x7A9D, 0x7A9D}, {0x7ACB, 0x7ACB}, {0x7AD9, 0x7AD9}, {0x7ADE, 0x7AE0}, {0x7AE5, 0x7AE5}, {0x7AEF, 0x7AEF},
    {0x7AF9, 0x7AF9}, {0x7B11, 0x7B11}, {0x7B14, 0x7B14}, {0x7B1B, 0x7B1B}, {0x7B26, 0x7B26}, {0x7B28, 0x7B28},
    {0x7B2C, 0x7B2C}, {0x7B49, 0x7B49}, {0x7B4B, 0x7B4B}, {0x7B51, 0x7B51}, {0x7B54, 0x7B54}, {0x7B56, 0x7B56},
    {0x7B79, 0x7B79}, {0x7B7E, 0x7B7E}, {0x7B80, 0x7B80}, {0x7B97, 0x7B97}, {0x7BA1, 0x7BA1}, {0x7BAD, 0x7BAD},
    {0x7BB1, 0x7BB1}, {0x7BC7, 0x7BC7}, {0x7BEE, 0x7BEE}, {0x7C3F, 0x7C3F}, {0x7C4D, 0x7C4D}, {0x7C73, 0x7C73},
    {0x7C7B, 0x7C7B}, {0x7C89, 0x7C89}, {0x7C92, 0x7C92}, {0x7C97, 0x7C97}, {0x7CA4, 0x7CA4}, {0x7CB9, 0x7CB9},
    {0x7CBE, 0x7CBE}, {0x7CCA, 0x7CCA}, {0x7CD5, 0x7CD6}, {0x7CDF, 0x7CDF}, {0x7CFB, 0x7CFB}, {0x7D20, 0x7D20},
    {0x7D22, 0x7D22}, {0x7D27, 0x7D27}, {0x7D2B, 0x7D2B}, {0x7D2F, 0x7D2F}, {0x7E41, 0x7E41}, {0x7EA2, 0x7EA2},
    {0x7EA6, 0x7EA7}, {0x7EAA, 0x7EAA}, {0x7EAF, 0x7EAF}, {0x7EB2, 0x7EB3}, {0x7EB5, 0x7EB5}, {0x7EB7, 0x7EB8},
    {0x7EBD, 0x7EBD}, {0x7EBF, 0x7EBF}, {0x7EC3, 0x7EC4}, {0x7EC6, 0x7EC8}, {0x7ECD, 0x7ECD}, {0x7ECF, 0x7ECF},
    {0x7ED3, 0x7ED3}, {0x7ED5, 0x7ED5}, {0x7ED8, 0x7ED9}, {0x7EDC, 0x7EDD}, {0x7EDF, 0x7EDF}, {0x7EE7, 0x7EE7},
    {0x7EE9, 0x7EEA}, {0x7EED, 0x7EED}, {0x7EF4, 0x7EF5}, {0x7EFC, 0x7EFC}, {0x7EFF, 0x7EFF}, {0x7F05, 0x7F05},
    {0x7F13, 0x7F13}, {0x7F16, 0x7F16}, {0x7F18, 0x7F18}, {0x7F20, 0x7F20}, {0x7F29, 0x7F29}, {0x7F34, 0x7F34},
    {0x7F36, 0x7F36}, {0x7F38, 0x7F38}, {0x7F3A, 0x7F3A}, {0x7F50, 0x7F51}, {0x7F55, 0x7F55}, {0x7F57, 0x7F57},
    {0x7F5A, 0x7F5A}, {0x7F62, 0x7F62}, {0x7F6A, 0x7F6A}, {0x7F6E, 0x7F6E}, {0x7F72, 0x7F72}, {0x7F8A, 0x7F8A},
    {0x7F8E, 0x7F8E}, {0x7F9E, 0x7F9E}, {0x7FA4, 0x7FA4}, {0x7FAF, 0x7FAF}, {0x7FBD, 0x7FBD}, {0x7FC1, 0x7FC1},
    {0x7FC5, 0x7FC5}, {0x7FD4, 0x7FD4}, {0x7FD8, 0x7FD8}, {0x7FE0, 0x7FE0}, {0x7FF0, 0x7FF0}, {0x7FFB, 0x7FFC},
    {0x8000, 0x8001}, {0x8003, 0x8003}, {0x8005, 0x8005}, {0x800C, 0x800D}, {0x8010, 0x8010}, {0x8017, 0x8017},
    {0x8033, 0x8033}, {0x8036, 0x8036}, {0x804A, 0x804A}, {0x804C, 0x804C}, {0x8054, 0x8054}, {0x8058, 0x8058},
    {0x805A, 0x805A}, {0x806A, 0x806A}, {0x8089, 0x8089}, {0x8096, 0x8096}, {0x809A, 0x809A}, {0x80A1, 0x80A1},
    {0x80A4, 0x80A5}, {0x80A9, 0x80A9}, {0x80AF, 0x80AF}, {0x80B2, 0x80B2}, {0x80C1, 0x80C1}, {0x80C6, 0x80C6},
    {0x80CC, 0x80CC}, {0x80CE, 0x80CE}, {0x80D6, 0x80D6}, {0x80DC, 0x80DC}, {0x80DE, 0x80DE}, {0x80E1, 0x80E1},
    {0x80F6, 0x80F6}, {0x80F8, 0x80F8}, {0x80FD, 0x80FD}, {0x8106, 0x8106}, {0x8111, 0x8111}, {0x8131, 0x8131},
    {0x8138, 0x8138}, {0x814A, 0x814A}, {0x8150, 0x8150}, {0x8153, 0x8153}, {0x8170, 0x8170}, {0x8179, 0x8179},
    {0x817E, 0x817F}, {0x81C2, 0x81C2}, {0x81E3, 0x81E3}, {0x81EA, 0x81EA}, {0x81ED, 0x81ED}, {0x81F3, 0x81F4},
    {0x820C, 0x820D}, {0x8212, 0x8212}, {0x821E, 0x821F}, {0x822A, 0x822A}, {0x822C, 0x822C}, {0x8230, 0x8230},
    {0x8239, 0x8239}, {0x826F, 0x826F}, {0x8272, 0x8272}, {0x827A, 0x827A}, {0x827E, 0x827E}, {0x8282, 0x8282},
    {0x8292, 0x8292}, {0x829D, 0x829D}, {0x82A6, 0x82A6}, {0x82AC, 0x82AD}, {0x82B1, 0x82B1}, {0x82B3, 0x82B3},
    {0x82CD, 0x82CD}, {0x82CF, 0x82CF}, {0x82D7, 0x82D7}, {0x82E5, 0x82E6}, {0x82F1, 0x82F1}, {0x8302, 0x8303},
    {0x8328, 0x8328}, {0x832B, 0x832B}, {0x8336, 0x8336}, {0x8349, 0x8349}, {0x8350, 0x8350}, {0x8352, 0x8352},
    {0x8363, 0x8363}, {0x836F, 0x836F}, {0x8377, 0x8377}, {0x8389, 0x8389}, {0x838E, 0x838E}, {0x83AA, 0x83AB},
    {0x83B1, 0x83B2}, {0x83B7, 0x83B7}, {0x83DC, 0x83DC}, {0x83E9, 0x83E9}, {0x83F2, 0x83F2}, {0x8404, 0x8404},
    {0x840D, 0x840D}, {0x8424, 0x8425}, {0x8427, 0x8428}, {0x843D, 0x843D}, {0x8457, 0x8457}, {0x845B, 0x845B},
    {0x8461, 0x8461}, {0x8482, 0x8482}, {0x848B, 0x848B}, {0x8499, 0x8499}, {0x84C9, 0x84C9}, {0x84DD, 0x84DD},
    {0x84EC, 0x84EC}, {0x8511, 0x8511}, {0x8521, 0x8521}, {0x8584, 0x8584}, {0x85AA, 0x85AA}, {0x85C9, 0x85C9},
    {0x85CF, 0x85CF}, {0x85E4, 0x85E4}, {0x864E, 0x864E}, {0x8651, 0x8651}, {0x866B, 0x866B}, {0x8679, 0x8679},
    {0x867D, 0x867E}, {0x8681, 0x8681}, {0x86C7, 0x86C7}, {0x86CB, 0x86CB}, {0x86D9, 0x86D9}, {0x86EE, 0x86EE},
    {0x8702, 0x8702}, {0x871C, 0x871C}, {0x8776, 0x8776}, {0x878D, 0x878D}, {0x87F9, 0x87F9}, {0x8822, 0x8822},
    {0x8840, 0x8840}, {0x884C, 0x884C}, {0x8857, 0x8857}, {0x8861, 0x8861}, {0x8863, 0x8863}, {0x8865, 0x8865},
    {0x8868, 0x8868}, {0x888B, 0x888B}, {0x88AB, 0x88AB}, {0x88AD, 0x88AD}, {0x88C1, 0x88C2}, {0x88C5, 0x88C5},
    {0x88D5, 0x88D5}, {0x88E4, 0x88E4}, {0x897F, 0x897F}, {0x8981, 0x8981}, {0x8986, 0x8986}, {0x89C1, 0x89C2},
    {0x89C4, 0x89C4}, {0x89C6, 0x89C6}, {0x89C8, 0x89C9}, {0x89D2, 0x89D2}, {0x89E3, 0x89E3}, {0x8A00, 0x8A00},
    {0x8A89, 0x8A89}, {0x8A93, 0x8A93}, {0x8B66, 0x8B66}, {0x8BA1, 0x8BA2}, {0x8BA4, 0x8BA4}, {0x8BA8, 0x8BA9},
    {0x8BAD, 0x8BB0}, {0x8BB2, 0x8BB2}, {0x8BB7, 0x8BB8}, {0x8BBA, 0x8BBA}, {0x8BBE, 0x8BBF}, {0x8BC1, 0x8BC1},
    {0x8BC4, 0x8BC4}, {0x8BC6, 0x8BC6}, {0x8BC9, 0x8BC9}, {0x8BCD, 0x8BCD}, {0x8BD1, 0x8BD1}, {0x8BD5, 0x8BD5},
    {0x8BD7, 0x8BD7}, {0x8BDA, 0x8BDA}, {0x8BDD, 0x8BDE}, {0x8BE2, 0x8BE2}, {0x8BE5, 0x8BE6}, {0x8BED, 0x8BED},
    {0x8BEF, 0x8BEF}, {0x8BF4, 0x8BF4}, {0x8BF7, 0x8BF8}, {0x8BFA, 0x8BFB}, {0x8BFE, 0x8BFE}, {0x8C01, 0x8C01},
    {0x8C03, 0x8C03}, {0x8C05, 0x8C05}, {0x8C08, 0x8C08}, {0x8C0A, 0x8C0B}, {0x8C13, 0x8C13}, {0x8C1C, 0x8C1C},
    {0x8C22, 0x8C22}, {0x8C28, 0x8C28}, {0x8C31, 0x8C31}, {0x8C37, 0x8C37}, {0x8C46, 0x8C46}, {0x8C61, 0x8C61},
    {0x8C6A, 0x8C6A}, {0x8C8C, 0x8C8C}, {0x8D1D, 0x8D1F}, {0x8D21, 0x8D25}, {0x8D27, 0x8D2A}, {0x8D2D, 0x8D2D},
    {0x8D2F, 0x8D2F}, {0x8D31, 0x8D31}, {0x8D34, 0x8D35}, {0x8D38, 0x8D3A}, {0x8D3C, 0x8D3C}, {0x8D3E, 0x8D3E},
    {0x8D44, 0x8D44}, {0x8D4B, 0x8D4C}, {0x8D4F, 0x8D50}, {0x8D54, 0x8D54}, {0x8D56, 0x8D56}, {0x8D5A, 0x8D5B},
    {0x8D5E, 0x8D5E}, {0x8D60, 0x8D60}, {0x8D62, 0x8D62}, {0x8D64, 0x8D64}, {0x8D6B, 0x8D6B}, {0x8D70, 0x8D70},
    {0x8D75, 0x8D75}, {0x8D77, 0x8D77}, {0x8D81, 0x8D81}, {0x8D85, 0x8D85}, {0x8D8A, 0x8D8B}, {0x8DA3, 0x8DA3},
    {0x8DB3, 0x8DB3}, {0x8DC3, 0x8DC3}, {0x8DCC, 0x8DCC}, {0x8DD1, 0x8DD1}, {0x8DDD, 0x8DDD}, {0x8DDF, 0x8DDF},
    {0x8DEF, 0x8DEF}, {0x8DF3, 0x8DF3}, {0x8E0F, 0x8E0F}, {0x8E22, 0x8E22}, {0x8E29, 0x8E29}, {0x8EAB, 0x8EAB},
    {0x8EB2, 0x8EB2}, {0x8F66, 0x8F66}, {0x8F68, 0x8F69}, {0x8F6C, 0x8F6C}, {0x8F6E, 0x8F70}, {0x8F7B, 0x8F7B},
    {0x8F7D, 0x8F7D}, {0x8F83, 0x8F83}, {0x8F85, 0x8F86}, {0x8F88, 0x8F89}, {0x8F91, 0x8F91}, {0x8F93, 0x8F93},
    {0x8F9B, 0x8F9B}, {0x8F9E, 0x8F9E}, {0x8FA8, 0x8FA9}, {0x8FB0, 0x8FB1}, {0x8FB9, 0x8FB9}, {0x8FBE, 0x8FBE},
    {0x8FC1, 0x8FC1}, {0x8FC5, 0x8FC5}, {0x8FC7, 0x8FC8}, {0x8FCE, 0x8FCE}, {0x8FD0, 0x8FD1}, {0x8FD4, 0x8FD4},
    {0x8FD8, 0x8FD9}, {0x8FDB, 0x8FDF}, {0x8FE6, 0x8FE6}, {0x8FEA, 0x8FEB}, {0x8FF0, 0x8FF0}, {0x8FF7, 0x8FF7},
    {0x8FFD, 0x8FFD}, {0x9000, 0x9003}, {0x9006, 0x9006}, {0x9009, 0x900A}, {0x900F, 0x9010}, {0x9012, 0x9012},
    {0x9014, 0x9014}, {0x901A, 0x901B}, {0x901D, 0x901D}, {0x901F, 0x9020}, {0x9022, 0x9022}, {0x9038, 0x9038},
    {0x903B, 0x903C}, {0x9047, 0x9047}, {0x904D, 0x904D}, {0x9053, 0x9053}, {0x9057, 0x9057}, {0x906D, 0x906E},
    {0x9075, 0x9075}, {0x907F, 0x9080}, {0x9093, 0x9093}, {0x90A3, 0x90A3}, {0x90A6, 0x90A6}, {0x90AA, 0x90AA},
    {0x90AE, 0x90AE}, {0x90B1, 0x90B1}, {0x90BB, 0x90BB}, {0x90CE, 0x90CE}, {0x90D1, 0x90D1}, {0x90E8, 0x90E8},
    {0x90ED, 0x90ED}, {0x90FD, 0x90FD}, {0x9102, 0x9102}, {0x9149, 0x9149}, {0x914B, 0x914B}, {0x914D, 0x914D},
    {0x9152, 0x9152}, {0x9177, 0x9178}, {0x9189, 0x9189}, {0x9192, 0x9192}, {0x91C7, 0x91C7}, {0x91CA, 0x91CA},
    {0x91CC, 0x91CF}, {0x91D1, 0x91D1}, {0x9488, 0x9488}, {0x9493, 0x9493}, {0x949F, 0x949F}, {0x94A2, 0x94A2},
    {0x94A6, 0x94A6}, {0x94B1, 0x94B1}, {0x94BB, 0x94BB}, {0x94C1, 0x94C1}, {0x94C3, 0x94C3}, {0x94DC, 0x94DC},
    {0x94E2, 0x94E2}, {0x94ED, 0x94ED}, {0x94F6, 0x94F6}, {0x94FA, 0x94FA}, {0x94FE, 0x94FE}, {0x9500, 0x9501},
    {0x9505, 0x9505}, {0x950B, 0x950B}, {0x9519, 0x9519}, {0x9521, 0x9521}, {0x9526, 0x9526}, {0x952E, 0x952E},
    {0x953A, 0x953A}, {0x9547, 0x9547}, {0x955C, 0x955C}, {0x956D, 0x956D}, {0x957F, 0x957F}, {0x95E8, 0x95E8},
    {0x95EA, 0x95EA}, {0x95ED, 0x95EE}, {0x95F0, 0x95F0}, {0x95F2, 0x95F2}, {0x95F4, 0x95F4}, {0x95F7, 0x95F7},
    {0x95F9, 0x95F9}, {0x95FB, 0x95FB}, {0x9601, 0x9601}, {0x9605, 0x9605}, {0x9610, 0x9610}, {0x9614, 0x9614},
    {0x961F, 0x961F}, {0x962E, 0x962E}, {0x9632, 0x9636}, {0x963B, 0x963B}, {0x963F, 0x9640}, {0x9644, 0x9646},
    {0x9648, 0x9648}, {0x964D, 0x964D}, {0x9650, 0x9650}, {0x9662, 0x9662}, {0x9664, 0x9664}, {0x9669, 0x966A},
    {0x9675, 0x9677}, {0x9686, 0x9686}, {0x968F, 0x9690}, {0x9694, 0x9694}, {0x969C, 0x969C}, {0x96BE, 0x96BE},
    {0x96C4, 0x96C6}, {0x96C9, 0x96C9}, {0x96E8, 0x96E8}, {0x96EA, 0x96EA}, {0x96EF, 0x96EF}, {0x96F3, 0x96F3},
    {0x96F6, 0x96F7}, {0x96FE, 0x96FE}, {0x9700, 0x9700}, {0x9707, 0x9707}, {0x970D, 0x970D}, {0x9716, 0x9716},
    {0x9732, 0x9732}, {0x9738, 0x9739}, {0x9752, 0x9752}, {0x9756, 0x9756}, {0x9759, 0x9759}, {0x975E, 0x975E},
    {0x9760, 0x9760}, {0x9762, 0x9762}, {0x9769, 0x9769}, {0x977C, 0x977C}, {0x978B, 0x978B}, {0x9791, 0x9791},
    {0x97E6, 0x97E6}, {0x97E9, 0x97E9}, {0x97F3, 0x97F3}, {0x9875, 0x9876}, {0x9879, 0x987B}, {0x987D, 0x987F},
    {0x9884, 0x9884}, {0x9886, 0x9887}, {0x9891, 0x9891}, {0x9897, 0x9898}, {0x989D, 0x989D}, {0x98CE, 0x98CE},
    {0x98D8, 0x98D9}, {0x98DE, 0x98DF}, {0x9910, 0x9910}, {0x996D, 0x996E}, {0x9970, 0x9971}, {0x997C, 0x997C},
    {0x9986, 0x9986}, {0x9996, 0x9996}, {0x9999, 0x9999}, {0x99A8, 0x99A8}, {0x9A6C, 0x9A6C}, {0x9A71, 0x9A71},
    {0x9A76, 0x9A76}, {0x9A7B, 0x9A7B}, {0x9A7E, 0x9A7E}, {0x9A8C, 0x9A8C}, {0x9A91, 0x9A91}, {0x9A97, 0x9A97},
    {0x9A9A, 0x9A9A}, {0x9AA4, 0x9AA4}, {0x9AA8, 0x9AA8}, {0x9AD8, 0x9AD8}, {0x9B3C, 0x9B3C}, {0x9B42, 0x9B42},
    {0x9B45, 0x9B45}, {0x9B54, 0x9B54}, {0x9C7C, 0x9C7C}, {0x9C81, 0x9C81}, {0x9C9C, 0x9C9C}, {0x9E1F, 0x9E1F},
    {0x9E21, 0x9E21}, {0x9E23, 0x9E23}, {0x9E2D, 0x9E2D}, {0x9E3F, 0x9E3F}, {0x9E45, 0x9E45}, {0x9E64, 0x9E64},
    {0x9E70, 0x9E70}, {0x9E7F, 0x9E7F}, {0x9EA6, 0x9EA6}, {0x9EBB, 0x9EBB}, {0x9EC4, 0x9EC4}, {0x9ECE, 0x9ECE},
    {0x9ED1, 0x9ED1}, {0x9ED8, 0x9ED8}, {0x9F13, 0x9F13}, {0x9F20, 0x9F20}, {0x9F3B, 0x9F3B}, {0x9F50, 0x9F50},
    {0x9F7F, 0x9F7F}, {0x9F84, 0x9F84}, {0x9F99, 0x9F99}, {0x9F9F, 0x9F9F}, {0xFE30, 0xFE31}, {0xFE33, 0xFE44},
    {0xFE49, 0xFE52}, {0xFE54, 0xFE57}, {0xFE59, 0xFE61}, {0xFE63, 0xFE63}, {0xFE68, 0xFE68}, {0xFE6A, 0xFE6B},
    {0xFF01, 0xFF03}, {0xFF05, 0xFF0A}, {0xFF0C, 0xFF0F}, {0xFF1A, 0xFF1B}, {0xFF1F, 0xFF20}, {0xFF3B, 0xFF3D},
    {0xFF3F, 0xFF3F}, {0xFF5B, 0xFF5B}, {0xFF5D, 0xFF5D},
}};

constexpr std::array<Range, 1898> traditionalChineseWritingSystemRanges = {{
    {0x0020, 0x0040}, {0x005B, 0x005F}, {0x007B, 0x007D}, {0x00A0, 0x00A0}, {0x00A7, 0x00A7}, {0x00A9, 0x00A9},
    {0x00B7, 0x00B7}, {0x2010, 0x2011}, {0x2018, 0x2019}, {0x2020, 0x2021}, {0x2025, 0x2027}, {0x2030, 0x2030},
    {0x2032, 0x2033}, {0x2035, 0x2035}, {0x203B, 0x203B}, {0x203E, 0x203E}, {0x3001, 0x3003}, {0x3007, 0x3011},
    {0x3014, 0x3015}, {0x301D, 0x301E}, {0x4E00, 0x4E01}, {0x4E03, 0x4E03}, {0x4E08, 0x4E0D}, {0x4E11, 0x4E11},
    {0x4E14, 0x4E14}, {0x4E16, 0x4E16}, {0x4E18, 0x4E19}, {0x4E1F, 0x4E1F}, {0x4E26, 0x4E26}, {0x4E28, 0x4E28},
    {0x4E2D, 0x4E2D}, {0x4E32, 0x4E32}, {0x4E36, 0x4E36}, {0x4E38, 0x4E39}, {0x4E3B, 0x4E3B}, {0x4E3F, 0x4E3F},
    {0x4E43, 0x4E43}, {0x4E45, 0x4E45}, {0x4E48, 0x4E48}, {0x4E4B, 0x4E4B}, {0x4E4E, 0x4E4F}, {0x4E56, 0x4E56},
    {0x4E58, 0x4E59}, {0x4E5D, 0x4E5D}, {0x4E5F, 0x4E5F}, {0x4E7E, 0x4E7E}, {0x4E82, 0x4E82}, {0x4E85, 0x4E86},
    {0x4E88, 0x4E88}, {0x4E8B, 0x4E8C}, {0x4E8E, 0x4E8E}, {0x4E91, 0x4E92}, {0x4E94, 0x4E95}, {0x4E9B, 0x4E9B},
    {0x4E9E, 0x4E9E}, {0x4EA0, 0x4EA1}, {0x4EA4, 0x4EA6}, {0x4EA8, 0x4EA8}, {0x4EAB, 0x4EAC}, {0x4EAE, 0x4EAE},
    {0x4EBA, 0x4EBA}, {0x4EC0, 0x4EC1}, {0x4EC7, 0x4EC7}, {0x4ECA, 0x4ECB}, {0x4ECD, 0x4ECD}, {0x4ED4, 0x4ED4},
    {0x4ED6, 0x4ED6}, {0x4ED8, 0x4ED9}, {0x4EE3, 0x4EE5}, {0x4EF0, 0x4EF0}, {0x4EF2, 0x4EF2}, {0x4EF6, 0x4EF6},
    {0x4EFB, 0x4EFB}, {0x4EFD, 0x4EFD}, {0x4F01, 0x4F01}, {0x4F0A, 0x4F0A}, {0x4F0D, 0x4F0D}, {0x4F10, 0x4F11},
    {0x4F19, 0x4F19}, {0x4F2F, 0x4F30}, {0x4F34, 0x4F34}, {0x4F38, 0x4F38}, {0x4F3C, 0x4F3D}, {0x4F46, 0x4F46},
    {0x4F48, 0x4F49}, {0x4F4D, 0x4F4F}, {0x4F54, 0x4F55}, {0x4F59, 0x4F59}, {0x4F5B, 0x4F5C}, {0x4F60, 0x4F60},
    {0x4F69, 0x4F69}, {0x4F73, 0x4F73}, {0x4F7F, 0x4F7F}, {0x4F86, 0x4F86}, {0x4F8B, 0x4F8B}, {0x4F9B, 0x4F9B},
    {0x4F9D, 0x4F9D}, {0x4FAF, 0x4FAF}, {0x4FB5, 0x4FB5}, {0x4FBF, 0x4FBF}, {0x4FC2, 0x4FC4}, {0x4FCA, 0x4FCA},
    {0x4FD7, 0x4FD7}, {0x4FDD, 0x4FDD}, {0x4FE0, 0x4FE1}, {0x4FEE, 0x4FEE}, {0x4FF1, 0x4FF1}, {0x4FFE, 0x4FFE},
    {0x500B, 0x500B}, {0x500D, 0x500D}, {0x5011, 0x5012}, {0x5019, 0x501A}, {0x501F, 0x501F}, {0x502B, 0x502B},
    {0x503C, 0x503C}, {0x5047, 0x5047}, {0x5049, 0x5049}, {0x504F, 0x504F}, {0x505A, 0x505A}, {0x505C, 0x505C},
    {0x5065, 0x5065}, {0x5074, 0x5077}, {0x5091, 0x5091}, {0x5099, 0x5099}, {0x50A2, 0x50A3}, {0x50B2, 0x50B3},
    {0x50B7, 0x50B7}, {0x50BB, 0x50BB}, {0x50BE, 0x50BE}, {0x50C5, 0x50C5}, {0x50CF, 0x50CF}, {0x50D1, 0x50D1},
    {0x50E7, 0x50E7}, {0x50F9, 0x50F9}, {0x5100, 0x5100}, {0x5104, 0x5104}, {0x5112, 0x5112}, {0x5118, 0x5118},
    {0x512A, 0x512A}, {0x513F, 0x513F}, {0x5141, 0x5141}, {0x5143, 0x5145}, {0x5147, 0x5149}, {0x514B, 0x514B},
    {0x514D, 0x514D}, {0x5152, 0x5152}, {0x5154, 0x5154}, {0x5165, 0x5165}, {0x5167, 0x5169}, {0x516B, 0x516E},
    {0x5171, 0x5171}, {0x5175, 0x5178}, {0x517C, 0x517C}, {0x5182, 0x5182}, {0x518A, 0x518A}, {0x518D, 0x518D},
    {0x5192, 0x5192}, {0x5196, 0x5196}, {0x51A0, 0x51A0}, {0x51AB, 0x51AC}, {0x51B0, 0x51B0}, {0x51B7, 0x51B7},
    {0x51C6, 0x51C6}, {0x51CC, 0x51CC}, {0x51DD, 0x51DD}, {0x51E0, 0x51E1}, {0x51F0, 0x51F1}, {0x51F5, 0x51F5},
    {0x51FA, 0x51FA}, {0x51FD, 0x51FD}, {0x5200, 0x5200}, {0x5206, 0x5207}, {0x520A, 0x520A}, {0x5217, 0x5217},
    {0x521D, 0x521D}, {0x5224, 0x5225}, {0x5229, 0x522A}, {0x5230, 0x5230}, {0x5236, 0x5237}, {0x523A, 0x523B},
    {0x5247, 0x5247}, {0x524C, 0x524D}, {0x525B, 0x525B}, {0x5269, 0x526A}, {0x526F, 0x526F}, {0x5272, 0x5272},
    {0x5275, 0x5275}, {0x5283, 0x5283}, {0x5287, 0x5287}, {0x5289, 0x5289}, {0x528D, 0x528D}, {0x529B, 0x529B},
    {0x529F, 0x52A0}, {0x52A9, 0x52AB}, {0x52C1, 0x52C1}, {0x52C7, 0x52C7}, {0x52C9, 0x52C9}, {0x52D2, 0x52D2},
    {0x52D5, 0x52D5}, {0x52D9, 0x52D9}, {0x52DD, 0x52DE}, {0x52E2, 0x52E2}, {0x52E4, 0x52E4}, {0x52F5, 0x52F5},
    {0x52F8, 0x52F9}, {0x52FF, 0x52FF}, {0x5305, 0x5305}, {0x5308, 0x5308}, {0x5315, 0x5317}, {0x531A, 0x531A},
    {0x5338, 0x5339}, {0x5340, 0x5341}, {0x5343, 0x5343}, {0x5347, 0x5348}, {0x534A, 0x534A}, {0x5352, 0x5354},
    {0x5357, 0x5357}, {0x535A, 0x535A}, {0x535C, 0x535C}, {0x5361, 0x5361}, {0x5369, 0x5369}, {0x536F, 0x5371},
    {0x5373, 0x5373}, {0x5377, 0x5377}, {0x537B, 0x537B}, {0x5382, 0x5382}, {0x5384, 0x5384}, {0x5398, 0x5398},
    {0x539A, 0x539A}, {0x539F, 0x539F}, {0x53AD, 0x53AD}, {0x53B2, 0x53B2}, {0x53B6, 0x53B6}, {0x53BB, 0x53BB},
    {0x53C3, 0x53C3}, {0x53C8, 0x53C8}, {0x53CA, 0x53CB}, {0x53CD, 0x53CD}, {0x53D4, 0x53D4}, {0x53D6, 0x53D7},
    {0x53E3, 0x53E6}, {0x53EA, 0x53ED}, {0x53EF, 0x53F0}, {0x53F2, 0x53F3}, {0x53F8, 0x53F8}, {0x5403, 0x5404},
    {0x5408, 0x540A}, {0x540C, 0x540E}, {0x5410, 0x5412}, {0x541B, 0x541B}, {0x541D, 0x5420}, {0x5426, 0x5427},
    {0x542B, 0x542B}, {0x5433, 0x5433}, {0x5435, 0x5435}, {0x5438, 0x5439}, {0x543E, 0x543E}, {0x5440, 0x5440},
    {0x5442, 0x5442}, {0x5446, 0x5446}, {0x544A, 0x544A}, {0x5462, 0x5462}, {0x5468, 0x5468}, {0x5473, 0x5473},
    {0x5475, 0x5475}, {0x547C, 0x547D}, {0x548C, 0x548C}, {0x5496, 0x5496}, {0x54A6, 0x54A7}, {0x54AA, 0x54AA},
    {0x54AC, 0x54AC}, {0x54B1, 0x54B1}, {0x54C0, 0x54C1}, {0x54C7, 0x54C9}, {0x54CE, 0x54CE}, {0x54E1, 0x54E1},
    {0x54E5, 0x54E6}, {0x54E9, 0x54EA}, {0x54ED, 0x54ED}, {0x54F2, 0x54F2}, {0x5509, 0x5509}, {0x5510, 0x5510},
    {0x5514, 0x5514}, {0x552C, 0x552C}, {0x552E, 0x552F}, {0x5531, 0x5531}, {0x5537, 0x5538}, {0x5546, 0x5546},
    {0x554A, 0x554A}, {0x554F, 0x554F}, {0x555F, 0x555F}, {0x5561, 0x5561}, {0x5565, 0x5566}, {0x556A, 0x556A},
    {0x5580, 0x5580}, {0x5582, 0x5582}, {0x5584, 0x5584}, {0x5587, 0x5587}, {0x558A, 0x558A}, {0x5594, 0x5594},
    {0x559C, 0x559D}, {0x55AC, 0x55AC}, {0x55AE, 0x55AE}, {0x55B5, 0x55B5}, {0x55CE, 0x55CE}, {0x55DA, 0x55DA},
    {0x55E8, 0x55E8}, {0x55EF, 0x55EF}, {0x5606, 0x5606}, {0x5609, 0x5609}, {0x5617, 0x5617}, {0x561B, 0x561B},
    {0x5634, 0x5634}, {0x563B, 0x563B}, {0x563F, 0x563F}, {0x5668, 0x5668}, {0x5674, 0x5674}, {0x5687, 0x5687},
    {0x56B4, 0x56B4}, {0x56C9, 0x56C9}, {0x56D7, 0x56D7}, {0x56DB, 0x56DB}, {0x56DE, 0x56DE}, {0x56E0, 0x56E0},
    {0x56F0, 0x56F0}, {0x56FA, 0x56FA}, {0x5708, 0x5708}, {0x570B, 0x570B}, {0x570D, 0x570D}, {0x5712, 0x5713},
    {0x5716, 0x5716}, {0x5718, 0x5718}, {0x571C, 0x571C}, {0x571F, 0x571F}, {0x5728, 0x5728}, {0x572D, 0x572D},
    {0x5730, 0x5730}, {0x573E, 0x573E}, {0x5740, 0x5740}, {0x5747, 0x5747}, {0x574E, 0x574E}, {0x5750, 0x5750},
    {0x5761, 0x5761}, {0x5764, 0x5764}, {0x5766, 0x5766}, {0x576A, 0x576A}, {0x5782, 0x5783}, {0x578B, 0x578B},
    {0x57C3, 0x57C3}, {0x57CE, 0x57CE}, {0x57D4, 0x57D4}, {0x57DF, 0x57DF}, {0x57F7, 0x57F7}, {0x57F9, 0x57FA},
    {0x5802, 0x5802}, {0x5805, 0x5806}, {0x5821, 0x5821}, {0x582A, 0x582A}, {0x5831, 0x5831}, {0x5834, 0x5834},
    {0x584A, 0x584A}, {0x5854, 0x5854}, {0x5857, 0x5857}, {0x585E, 0x585E}, {0x586B, 0x586B}, {0x5875, 0x5875},
    {0x5883, 0x5883}, {0x589E, 0x589E}, {0x58A8, 0x58A8}, {0x58AE, 0x58AE}, {0x58C1, 0x58C1}, {0x58C7, 0x58C7},
    {0x58D3, 0x58D3}, {0x58D8, 0x58D8}, {0x58DE, 0x58DE}, {0x58E2, 0x58E2}, {0x58EB, 0x58EC}, {0x58EF, 0x58EF},
    {0x58FD, 0x58FD}, {0x5902, 0x5902}, {0x590A, 0x590A}, {0x590F, 0x590F}, {0x5915, 0x5916}, {0x591A, 0x591A},
    {0x591C, 0x591C}, {0x5920, 0x5920}, {0x5922, 0x5922}, {0x5925, 0x5925}, {0x5927, 0x5927}, {0x5929, 0x592B},
    {0x592E, 0x592E}, {0x5931, 0x5931}, {0x5937, 0x5938}, {0x593E, 0x593E}, {0x5947, 0x5949}, {0x594E, 0x594F},
    {0x5951, 0x5951}, {0x5954, 0x5954}, {0x5957, 0x5957}, {0x5967, 0x5967}, {0x596A, 0x596A}, {0x596E, 0x596E},
    {0x5973, 0x5974}, {0x5976, 0x5976}, {0x5979, 0x5979}, {0x597D, 0x597D}, {0x5982, 0x5982}, {0x5999, 0x5999},
    {0x599D, 0x599D}, {0x59A5, 0x59A5}, {0x59A8, 0x59A8}, {0x59AE, 0x59AE}, {0x59B3, 0x59B3}, {0x59B9, 0x59B9},
    {0x59BB, 0x59BB}, {0x59C6, 0x59C6}, {0x59CA, 0x59CB}, {0x59D0, 0x59D1}, {0x59D3, 0x59D4}, {0x59FF, 0x59FF},
    {0x5A01, 0x5A01}, {0x5A03, 0x5A03}, {0x5A18, 0x5A18}, {0x5A1B, 0x5A1B}, {0x5A41, 0x5A41}, {0x5A46, 0x5A46},
    {0x5A5A, 0x5A5A}, {0x5A66, 0x5A66}, {0x5A92, 0x5A92}, {0x5ABD, 0x5ABD}, {0x5ACC, 0x5ACC}, {0x5AE9, 0x5AE9},
    {0x5B50, 0x5B50}, {0x5B54, 0x5B54}, {0x5B57, 0x5B58}, {0x5B5D, 0x5B5D}, {0x5B5F, 0x5B5F}, {0x5B63, 0x5B64},
    {0x5B69, 0x5B69}, {0x5B6B, 0x5B6B}, {0x5B78, 0x5B78}, {0x5B80, 0x5B80}, {0x5B83, 0x5B83}, {0x5B85, 0x5B85},
    {0x5B87, 0x5B89}, {0x5B8B, 0x5B8C}, {0x5B8F, 0x5B8F}, {0x5B97, 0x5B9C}, {0x5BA2, 0x5BA4}, {0x5BAE, 0x5BAE},
    {0x5BB3, 0x5BB3}, {0x5BB6, 0x5BB6}, {0x5BB9, 0x5BB9}, {0x5BBF, 0x5BBF}, {0x5BC2, 0x5BC2}, {0x5BC4, 0x5BC6},
    {0x5BCC, 0x5BCC}, {0x5BD2, 0x5BD2}, {0x5BDE, 0x5BDF}, {0x5BE2, 0x5BE2}, {0x5BE6, 0x5BE9}, {0x5BEB, 0x5BEC},
    {0x5BEE, 0x5BEE}, {0x5BF5, 0x5BF6}, {0x5BF8, 0x5BF8}, {0x5C01, 0x5C01}, {0x5C04, 0x5C04}, {0x5C07, 0x5C08},
    {0x5C0A, 0x5C0B}, {0x5C0D, 0x5C0F}, {0x5C11, 0x5C11}, {0x5C16, 0x5C16}, {0x5C1A, 0x5C1A}, {0x5C22, 0x5C22},
    {0x5C24, 0x5C24}, {0x5C31, 0x5C31}, {0x5C38, 0x5C38}, {0x5C3A, 0x5C3A}, {0x5C3C, 0x5C3C}, {0x5C3E, 0x5C3E},
    {0x5C40, 0x5C41}, {0x5C45, 0x5C46}, {0x5C4B, 0x5C4B}, {0x5C4F, 0x5C4F}, {0x5C55, 0x5C55}, {0x5C60, 0x5C60},
    {0x5C64, 0x5C64}, {0x5C6C, 0x5C6C}, {0x5C6E, 0x5C6E}, {0x5C71, 0x5C71}, {0x5CA1, 0x5CA1}, {0x5CA9, 0x5CA9},
    {0x5CB8, 0x5CB8}, {0x5CF0, 0x5CF0}, {0x5CF6, 0x5CF6}, {0x5CFD, 0x5CFD}, {0x5D07, 0x5D07}, {0x5D19, 0x5D19},
    {0x5D34, 0x5D34}, {0x5D50, 0x5D50}, {0x5DBA, 0x5DBA}, {0x5DDB, 0x5DDB}, {0x5DDD, 0x5DDE}, {0x5DE1, 0x5DE1},
    {0x5DE5, 0x5DE8}, {0x5DEB, 0x5DEB}, {0x5DEE, 0x5DEE}, {0x5DF1, 0x5DF4}, {0x5DF7, 0x5DF7}, {0x5DFE, 0x5DFE},
    {0x5E02, 0x5E03}, {0x5E0C, 0x5E0C}, {0x5E15, 0x5E16}, {0x5E1B, 0x5E1B}, {0x5E1D, 0x5E1D}, {0x5E25, 0x5E25},
    {0x5E2B, 0x5E2B}, {0x5E2D, 0x5E2D}, {0x5E33, 0x5E33}, {0x5E36, 0x5E36}, {0x5E38, 0x5E38}, {0x5E3D, 0x5E3D},
    {0x5E45, 0x5E45}, {0x5E55, 0x5E55}, {0x5E63, 0x5E63}, {0x5E6B, 0x5E6B}, {0x5E72, 0x5E74}, {0x5E78, 0x5E7F},
    {0x5E87, 0x5E87}, {0x5E8A, 0x5E8A}, {0x5E8F, 0x5E8F}, {0x5E95, 0x5E95}, {0x5E97, 0x5E97}, {0x5E9A, 0x5E9A},
    {0x5E9C, 0x5E9C}, {0x5EA6, 0x5EA7}, {0x5EAB, 0x5EAB}, {0x5EAD, 0x5EAD}, {0x5EB7, 0x5EB8}, {0x5EC9, 0x5EC9},
    {0x5ED6, 0x5ED6}, {0x5EE0, 0x5EE0}, {0x5EE2, 0x5EE3}, {0x5EF3, 0x5EF4}, {0x5EF6, 0x5EF7}, {0x5EFA, 0x5EFA},
    {0x5EFE, 0x5EFE}, {0x5F04, 0x5F04}, {0x5F0B, 0x5F0B}, {0x5F0F, 0x5F0F}, {0x5F13, 0x5F13}, {0x5F15, 0x5F15},
    {0x5F17, 0x5F18}, {0x5F1F, 0x5F1F}, {0x5F26, 0x5F26}, {0x5F31, 0x5F31}, {0x5F35, 0x5F35}, {0x5F37, 0x5F37},
    {0x5F48, 0x5F48}, {0x5F4A, 0x5F4A}, {0x5F4C, 0x5F4C}, {0x5F4E, 0x5F4E}, {0x5F50, 0x5F50}, {0x5F5D, 0x5F5E},
    {0x5F61, 0x5F62}, {0x5F65, 0x5F65}, {0x5F69, 0x5F69}, {0x5F6C, 0x5F6D}, {0x5F70, 0x5F71}, {0x5F73, 0x5F73},
    {0x5F79, 0x5F79}, {0x5F7C, 0x5F7C}, {0x5F80, 0x5F81}, {0x5F85, 0x5F85}, {0x5F88, 0x5F88}, {0x5F8B, 0x5F8C},
    {0x5F90, 0x5F92}, {0x5F97, 0x5F97}, {0x5F9E, 0x5F9E}, {0x5FA9, 0x5FA9}, {0x5FAE, 0x5FAE}, {0x5FB5, 0x5FB5},
    {0x5FB7, 0x5FB7}, {0x5FB9, 0x5FB9}, {0x5FC3, 0x5FC3}, {0x5FC5, 0x5FC5}, {0x5FCC, 0x5FCD}, {0x5FD7, 0x5FD9},
    {0x5FE0, 0x5FE0}, {0x5FEB, 0x5FEB}, {0x5FF5, 0x5FF5}, {0x5FFD, 0x5FFD}, {0x600E, 0x600E}, {0x6012, 0x6012},
    {0x6015, 0x6016}, {0x601D, 0x601D}, {0x6021, 0x6021}, {0x6025, 0x6025}, {0x6027, 0x6028}, {0x602A, 0x602A},
    {0x6046, 0x6046}, {0x6050, 0x6050}, {0x6062, 0x6062}, {0x6065, 0x6065}, {0x6068, 0x6069}, {0x606D, 0x606D},
    {0x606F, 0x6070}, {0x6085, 0x6085}, {0x6089, 0x6089}, {0x6094, 0x6094}, {0x609F, 0x60A0}, {0x60A8, 0x60A8},
    {0x60B2, 0x60B2}, {0x60B6, 0x60B6}, {0x60C5, 0x60C5}, {0x60D1, 0x60D1}, {0x60DC, 0x60DC}, {0x60E0, 0x60E1},
    {0x60F1, 0x60F1}, {0x60F3, 0x60F3}, {0x60F9, 0x60F9}, {0x6101, 0x6101}, {0x6108, 0x6109}, {0x610F, 0x610F},
    {0x611A, 0x611B}, {0x611F, 0x611F}, {0x6148, 0x6148}, {0x614B, 0x614B}, {0x6155, 0x6155}, {0x6158, 0x6158},
    {0x6162, 0x6163}, {0x6167, 0x6167}, {0x616E, 0x616E}, {0x6170, 0x6170}, {0x6176, 0x6176}, {0x617E, 0x617E},
    {0x6182, 0x6182}, {0x6190, 0x6191}, {0x61B2, 0x61B2}, {0x61B6, 0x61B6}, {0x61BE, 0x61BE}, {0x61C2, 0x61C2},
    {0x61C9, 0x61C9}, {0x61F6, 0x61F7}, {0x61FC, 0x61FC}, {0x6200, 0x6200}, {0x6208, 0x6208}, {0x620A, 0x620A},
    {0x620C, 0x620C}, {0x6210, 0x6212}, {0x6216, 0x6216}, {0x622A, 0x622A}, {0x6230, 0x6230}, {0x6232, 0x6232},
    {0x6234, 0x6234}, {0x6236, 0x6236}, {0x623F, 0x6241}, {0x6247, 0x6247}, {0x624B, 0x624B}, {0x624D, 0x624E},
    {0x6253, 0x6253}, {0x6258, 0x6258}, {0x6263, 0x6263}, {0x6265, 0x6265}, {0x626D, 0x626D}, {0x626F, 0x626F},
    {0x6279, 0x6279}, {0x627E, 0x6280}, {0x6284, 0x6284}, {0x628A, 0x628A}, {0x6293, 0x6293}, {0x6295, 0x6295},
    {0x6297, 0x6298}, {0x62AB, 0x62AC}, {0x62B1, 0x62B1}, {0x62B5, 0x62B5}, {0x62B9, 0x62B9}, {0x62BD, 0x62BD},
    {0x62C6, 0x62C6}, {0x62C9, 0x62C9}, {0x62CB, 0x62CB}, {0x62CD, 0x62CD}, {0x62CF, 0x62CF}, {0x62D2, 0x62D2},
    {0x62D4, 0x62D4}, {0x62D6, 0x62D6}, {0x62DB, 0x62DC}, {0x62EC, 0x62EC}, {0x62F3, 0x62F3}, {0x62FC, 0x62FC},
    {0x62FE, 0x62FF}, {0x6301, 0x6301}, {0x6307, 0x6307}, {0x6309, 0x6309}, {0x6311, 0x6311}, {0x6316, 0x6316},
    {0x632A, 0x632A}, {0x632F, 0x632F}, {0x633A, 0x633A}, {0x6350, 0x6350}, {0x6355, 0x6355}, {0x6368, 0x6368},
    {0x6372, 0x6372}, {0x6377, 0x6377}, {0x6383, 0x6383}, {0x6388, 0x6389}, {0x638C, 0x638C}, {0x6392, 0x6392},
    {0x639B, 0x639B}, {0x63A1, 0x63A2}, {0x63A5, 0x63A5}, {0x63A7, 0x63A8}, {0x63AA, 0x63AA}, {0x63CF, 0x63D0},
    {0x63D2, 0x63D2}, {0x63DA, 0x63DB}, {0x63E1, 0x63E1}, {0x63EE, 0x63EE}, {0x63F4, 0x63F4}, {0x640D, 0x640D},
    {0x6416, 0x6416}, {0x641C, 0x641C}, {0x641E, 0x641E}, {0x642C, 0x642D}, {0x6436, 0x6436}, {0x6458, 0x6458},
    {0x6469, 0x6469}, {0x6478, 0x6478}, {0x6490, 0x6490}, {0x6492, 0x6492}, {0x649E, 0x649E}, {0x64A3, 0x64A3},
    {0x64A5, 0x64A5}, {0x64AD, 0x64AD}, {0x64BE, 0x64BF}, {0x64C1, 0x64C1}, {0x64C7, 0x64C7}, {0x64CA, 0x64CB},
    {0x64CD, 0x64CE}, {0x64D4, 0x64D4}, {0x64DA, 0x64DA}, {0x64E0, 0x64E0}, {0x64E6, 0x64E6}, {0x64EC, 0x64EC},
    {0x64F4, 0x64F4}, {0x64FA, 0x64FA}, {0x64FE, 0x64FE}, {0x651D, 0x651D}, {0x652F, 0x652F}, {0x6534, 0x6534},
    {0x6536, 0x6536}, {0x6539, 0x6539}, {0x653B, 0x653B}, {0x653E, 0x653F}, {0x6545, 0x6545}, {0x6548, 0x6548},
    {0x654D, 0x654D}, {0x654F, 0x654F}, {0x6551, 0x6551}, {0x6557, 0x6559}, {0x655D, 0x655D}, {0x6562, 0x6563},
    {0x6566, 0x6566}, {0x656C, 0x656C}, {0x6574, 0x6575}, {0x6578, 0x6578}, {0x6587, 0x6587}, {0x6590, 0x6590},
    {0x6597, 0x6597}, {0x6599, 0x6599}, {0x65A4, 0x65A4}, {0x65AF, 0x65B0}, {0x65B7, 0x65B7}, {0x65B9, 0x65B9},
    {0x65BC, 0x65BD}, {0x65C1, 0x65C1}, {0x65C5, 0x65C5}, {0x65CB, 0x65CB}, {0x65CF, 0x65CF}, {0x65D7, 0x65D7},
    {0x65E0, 0x65E0}, {0x65E2, 0x65E2}, {0x65E5, 0x65E6}, {0x65E9, 0x65E9}, {0x65ED, 0x65ED}, {0x65FA, 0x65FA},
    {0x6602, 0x6602}, {0x6606, 0x6607}, {0x660C, 0x660C}, {0x660E, 0x660F}, {0x6613, 0x6613}, {0x661F, 0x6620},
    {0x6625, 0x6625}, {0x6628, 0x6628}, {0x662D, 0x662D}, {0x662F, 0x662F}, {0x6642, 0x6642}, {0x6649, 0x6649},
    {0x6652, 0x6652}, {0x665A, 0x665A}, {0x6668, 0x6668}, {0x666E, 0x666F}, {0x6674, 0x6674}, {0x6676, 0x6676},
    {0x667A, 0x667A}, {0x6691, 0x6691}, {0x6696, 0x6697}, {0x66AB, 0x66AB}, {0x66B4, 0x66B4}, {0x66C6, 0x66C6},
    {0x66C9, 0x66C9}, {0x66F0, 0x66F0}, {0x66F2, 0x66F2}, {0x66F4, 0x66F4}, {0x66F8, 0x66F8}, {0x66FC, 0x66FC},
    {0x66FE, 0x6700}, {0x6703, 0x6703}, {0x6708, 0x6709}, {0x670B, 0x670B}, {0x670D, 0x670D}, {0x6717, 0x6717},
    {0x671B, 0x671B}, {0x671D, 0x671D}, {0x671F, 0x671F}, {0x6728, 0x6728}, {0x672A, 0x672D}, {0x6731, 0x6731},
    {0x6735, 0x6735}, {0x6749, 0x6749}, {0x674E, 0x674E}, {0x6750, 0x6751}, {0x675C, 0x675C}, {0x675F, 0x675F},
    {0x676F, 0x6771}, {0x677E, 0x677F}, {0x6790, 0x6790}, {0x6797, 0x6797}, {0x679C, 0x679D}, {0x67B6, 0x67B6},
    {0x67CF, 0x67D0}, {0x67D3, 0x67D4}, {0x67E5, 0x67E5}, {0x67EC, 0x67EC}, {0x67EF, 0x67EF}, {0x67F3, 0x67F4},
    {0x6821, 0x6821}, {0x6838, 0x6839}, {0x683C, 0x683C}, {0x6843, 0x6843}, {0x6848, 0x6848}, {0x684C, 0x684C},
    {0x6851, 0x6851}, {0x6881, 0x6881}, {0x6885, 0x6885}, {0x689D, 0x689D}, {0x68A8, 0x68A8}, {0x68AF, 0x68B0},
    {0x68B5, 0x68B5}, {0x68C4, 0x68C4}, {0x68C9, 0x68C9}, {0x68CB, 0x68CB}, {0x68D2, 0x68D2}, {0x68DA, 0x68DA},
    {0x68EE, 0x68EE}, {0x6905, 0x6905}, {0x690D, 0x690D}, {0x6930, 0x6930}, {0x694A, 0x694A}, {0x6953, 0x6953},
    {0x695A, 0x695A}, {0x696D, 0x696D}, {0x6975, 0x6975}, {0x6982, 0x6982}, {0x699C, 0x699C}, {0x69AE, 0x69AE},
    {0x69CB, 0x69CB}, {0x69CD, 0x69CD}, {0x6A02, 0x6A02}, {0x6A13, 0x6A13}, {0x6A19, 0x6A19}, {0x6A1E, 0x6A1E},
    {0x6A21, 0x6A21}, {0x6A23, 0x6A23}, {0x6A39, 0x6A39}, {0x6A4B, 0x6A4B}, {0x6A5F, 0x6A5F}, {0x6A6B, 0x6A6B},
    {0x6A80, 0x6A80}, {0x6A94, 0x6A94}, {0x6AA2, 0x6AA2}, {0x6B04, 0x6B04}, {0x6B0A, 0x6B0A}, {0x6B20, 0x6B21},
    {0x6B23, 0x6B23}, {0x6B32, 0x6B32}, {0x6B3A, 0x6B3A}, {0x6B3D, 0x6B3E}, {0x6B49, 0x6B49}, {0x6B4C, 0x6B4C},
    {0x6B50, 0x6B50}, {0x6B61, 0x6B66}, {0x6B72, 0x6B72}, {0x6B77, 0x6B79}, {0x6B7B, 0x6B7B}, {0x6B8A, 0x6B8A},
    {0x6B98, 0x6B98}, {0x6BB3, 0x6BB3}, {0x6BB5, 0x6BB5}, {0x6BBA, 0x6BBA}, {0x6BBC, 0x6BBC}, {0x6BC0, 0x6BC0},
    {0x6BC5, 0x6BC5}, {0x6BCB, 0x6BCB}, {0x6BCD, 0x6BCD}, {0x6BCF, 0x6BCF}, {0x6BD2, 0x6BD2}, {0x6BD4, 0x6BD4},
    {0x6BDB, 0x6BDB}, {0x6BEB, 0x6BEB}, {0x6C0F, 0x6C0F}, {0x6C11, 0x6C11}, {0x6C14, 0x6C14}, {0x6C23, 0x6C23},
    {0x6C34, 0x6C34}, {0x6C38, 0x6C38}, {0x6C42, 0x6C42}, {0x6C57, 0x6C57}, {0x6C5D, 0x6C5D}, {0x6C5F, 0x6C61},
    {0x6C6A, 0x6C6A}, {0x6C76, 0x6C76}, {0x6C7A, 0x6C7A}, {0x6C7D, 0x6C7D}, {0x6C83, 0x6C83}, {0x6C88, 0x6C89},
    {0x6C92, 0x6C92}, {0x6C96, 0x6C96}, {0x6C99, 0x6C99}, {0x6CB3, 0x6CB3}, {0x6CB9, 0x6CB9}, {0x6CBB, 0x6CBB},
    {0x6CBF, 0x6CBF}, {0x6CC1, 0x6CC1}, {0x6CC9, 0x6CCA}, {0x6CD5, 0x6CD5}, {0x6CE1, 0x6CE2}, {0x6CE5, 0x6CE5},
    {0x6CE8, 0x6CE8}, {0x6CF0, 0x6CF0}, {0x6CF3, 0x6CF3}, {0x6D0B, 0x6D0B}, {0x6D17, 0x6D17}, {0x6D1B, 0x6D1B},
    {0x6D1E, 0x6D1E}, {0x6D29, 0x6D2A}, {0x6D32, 0x6D32}, {0x6D3B, 0x6D3B}, {0x6D3D, 0x6D3E}, {0x6D41, 0x6D41},
    {0x6D66, 0x6D66}, {0x6D69, 0x6D6A}, {0x6D6E, 0x6D6E}, {0x6D77, 0x6D77}, {0x6D87, 0x6D89}, {0x6DAF, 0x6DAF},
    {0x6DB2, 0x6DB2}, {0x6DB5, 0x6DB5}, {0x6DBC, 0x6DBC}, {0x6DD1, 0x6DD1}, {0x6DDA, 0x6DDA}, {0x6DE1, 0x6DE1},
    {0x6DE8, 0x6DE8}, {0x6DF1, 0x6DF1}, {0x6DF7, 0x6DF7}, {0x6DFA, 0x6DFA}, {0x6E05, 0x6E05}, {0x6E1B, 0x6E1B},
    {0x6E21, 0x6E21}, {0x6E2C, 0x6E2C}, {0x6E2F, 0x6E2F}, {0x6E38, 0x6E38}, {0x6E56, 0x6E56}, {0x6E6F, 0x6E6F},
    {0x6E90, 0x6E90}, {0x6E96, 0x6E96}, {0x6E9D, 0x6E9D}, {0x6EAA, 0x6EAB}, {0x6EC4, 0x6EC5}, {0x6ECB, 0x6ECB},
    {0x6ED1, 0x6ED1}, {0x6EF4, 0x6EF4}, {0x6EFE, 0x6EFF}, {0x6F02, 0x6F02}, {0x6F0F, 0x6F0F}, {0x6F14, 0x6F14},
    {0x6F20, 0x6F20}, {0x6F22, 0x6F22}, {0x6F2B, 0x6F2B}, {0x6F32, 0x6F32}, {0x6F38, 0x6F38}, {0x6F54, 0x6F54},
    {0x6F58, 0x6F58}, {0x6F5B, 0x6F5B}, {0x6F6E, 0x6F6E}, {0x6FA4, 0x6FA4}, {0x6FB3, 0x6FB3}, {0x6FC0, 0x6FC0},
    {0x6FC3, 0x6FC3}, {0x6FDF, 0x6FDF}, {0x6FE4, 0x6FE4}, {0x6FEB, 0x6FEB}, {0x6FF1, 0x6FF1}, {0x700F, 0x700F},
    {0x704C, 0x704C}, {0x7063, 0x7063}, {0x706B, 0x706B}, {0x7070, 0x7070}, {0x707D, 0x707D}, {0x708E, 0x708E},
    {0x70AE, 0x70AE}, {0x70B8, 0x70B8}, {0x70BA, 0x70BA}, {0x70C8, 0x70C8}, {0x70CF, 0x70CF}, {0x70E4, 0x70E4},
    {0x7121, 0x7121}, {0x7126, 0x7126}, {0x7136, 0x7136}, {0x7159, 0x7159}, {0x715E, 0x715E}, {0x7167, 0x7167},
    {0x7169, 0x7169}, {0x718A, 0x718A}, {0x719F, 0x719F}, {0x71B1, 0x71B1}, {0x71C3, 0x71C3}, {0x71C8, 0x71C8},
    {0x71D2, 0x71D2}, {0x71DF, 0x71DF}, {0x7206, 0x7206}, {0x7210, 0x7210}, {0x721B, 0x721B}, {0x722A, 0x722A},
    {0x722C, 0x722D}, {0x7235, 0x7236}, {0x7238, 0x7238}, {0x723A, 0x723B}, {0x723D, 0x723F}, {0x7246, 0x7248},
    {0x724C, 0x724C}, {0x7259, 0x7259}, {0x725B, 0x725B}, {0x7260, 0x7260}, {0x7267, 0x7267}, {0x7269, 0x7269},
    {0x7272, 0x7272}, {0x7279, 0x7279}, {0x727D, 0x727D}, {0x72A7, 0x72A7}, {0x72AC, 0x72AC}, {0x72AF, 0x72AF},
    {0x72C0, 0x72C0}, {0x72C2, 0x72C2}, {0x72D0, 0x72D0}, {0x72D7, 0x72D7}, {0x72E0, 0x72E0}, {0x72FC, 0x72FC},
    {0x731B, 0x731C}, {0x7334, 0x7334}, {0x7336, 0x7336}, {0x7344, 0x7345}, {0x734E, 0x734E}, {0x7368, 0x7368},
    {0x7372, 0x7372}, {0x7378, 0x7378}, {0x737B, 0x737B}, {0x7384, 0x7384}, {0x7387, 0x7387}, {0x7389, 0x7389},
    {0x738B, 0x738B}, {0x73A9, 0x73A9}, {0x73AB, 0x73AB}, {0x73B2, 0x73B2}, {0x73BB, 0x73BB}, {0x73CA, 0x73CA},
    {0x73CD, 0x73CD}, {0x73E0, 0x73E0}, {0x73E5, 0x73E5}, {0x73ED, 0x73ED}, {0x73FE, 0x73FE}, {0x7403, 0x7403},
    {0x7406, 0x7406}, {0x7409, 0x7409}, {0x742A, 0x742A}, {0x7434, 0x7434}, {0x7459, 0x7459}, {0x745C, 0x745C},
    {0x745E, 0x745F}, {0x7464, 0x7464}, {0x746A, 0x746A}, {0x7470, 0x7470}, {0x74B0, 0x74B0}, {0x74DC, 0x74DC},
    {0x74E6, 0x74E6}, {0x74F6, 0x74F6}, {0x7518, 0x7518}, {0x751A, 0x751A}, {0x751C, 0x751C}, {0x751F, 0x751F},
    {0x7522, 0x7522}, {0x7528, 0x7528}, {0x7530, 0x7533}, {0x7537, 0x7538}, {0x754C, 0x754C}, {0x7559, 0x7559},
    {0x7562, 0x7562}, {0x7565, 0x7565}, {0x756A, 0x756B}, {0x7570, 0x7570}, {0x7576, 0x7576}, {0x7586, 0x7586},
    {0x758B, 0x758B}, {0x758F, 0x758F}, {0x7591, 0x7592}, {0x75BC, 0x75BC}, {0x75C5, 0x75C5}, {0x75D5, 0x75D5},
    {0x75DB, 0x75DB}, {0x75F4, 0x75F4}, {0x760B, 0x760B}, {0x7642, 0x7642}, {0x7661, 0x7661}, {0x7676, 0x7676},
    {0x7678, 0x7678}, {0x767B, 0x767E}, {0x7684, 0x7684}, {0x7686, 0x7687}, {0x76AE, 0x76AE}, {0x76BF, 0x76BF},
    {0x76C3, 0x76C3}, {0x76CA, 0x76CA}, {0x76DB, 0x76DC}, {0x76DF, 0x76DF}, {0x76E1, 0x76E1}, {0x76E3, 0x76E4},
    {0x76E7, 0x76E7}, {0x76EE, 0x76EE}, {0x76F2, 0x76F2}, {0x76F4, 0x76F4}, {0x76F8, 0x76F8}, {0x76FC, 0x76FC},
    {0x76FE, 0x76FE}, {0x7701, 0x7701}, {0x7709, 0x7709}, {0x770B, 0x770B}, {0x771F, 0x7720}, {0x773C, 0x773C},
    {0x773E, 0x773E}, {0x775B, 0x775B}, {0x7761, 0x7761}, {0x7763, 0x7763}, {0x77A7, 0x77A7}, {0x77AD, 0x77AD},
    {0x77DB, 0x77DB}, {0x77E2, 0x77E3}, {0x77E5, 0x77E5}, {0x77ED, 0x77ED}, {0x77F3, 0x77F3}, {0x7802, 0x7802},
    {0x780D, 0x780D}, {0x7814, 0x7814}, {0x7832, 0x7832}, {0x7834, 0x7834}, {0x786C, 0x786C}, {0x788E, 0x788E},
    {0x7897, 0x7897}, {0x789F, 0x789F}, {0x78A7, 0x78A7}, {0x78A9, 0x78A9}, {0x78B0, 0x78B0}, {0x78BA, 0x78BA},
    {0x78BC, 0x78BC}, {0x78C1, 0x78C1}, {0x78E8, 0x78E8}, {0x78EF, 0x78EF}, {0x790E, 0x790E}, {0x7919, 0x7919},
    {0x793A, 0x793A}, {0x793E, 0x793E}, {0x7955, 0x7956}, {0x795A, 0x795B}, {0x795D, 0x795E}, {0x7965, 0x7965},
    {0x7968, 0x7968}, {0x797F, 0x797F}, {0x7981, 0x7981}, {0x798D, 0x798F}, {0x79AA, 0x79AA}, {0x79AE, 0x79AE},
    {0x79B8, 0x79B8}, {0x79BE, 0x79BE}, {0x79C0, 0x79C1}, {0x79CB, 0x79CB}, {0x79D1, 0x79D2}, {0x79D8, 0x79D8},
    {0x79DF, 0x79DF}, {0x79E4, 0x79E4}, {0x79E6, 0x79E6}, {0x79FB, 0x79FB}, {0x7A05, 0x7A05}, {0x7A0B, 0x7A0B},
    {0x7A0D, 0x7A0D}, {0x7A2E, 0x7A2E}, {0x7A31, 0x7A31}, {0x7A3F, 0x7A3F}, {0x7A46, 0x7A46}, {0x7A4C, 0x7A4D},
    {0x7A69, 0x7A69}, {0x7A74, 0x7A74}, {0x7A76, 0x7A76}, {0x7A79, 0x7A7A}, {0x7A7F, 0x7A7F}, {0x7A81, 0x7A81},
    {0x7A97, 0x7A97}, {0x7AA9, 0x7AA9}, {0x7AAE, 0x7AAE}, {0x7AB6, 0x7AB6}, {0x7ACB, 0x7ACB}, {0x7AD9, 0x7AD9},
    {0x7ADF, 0x7AE0}, {0x7AE5, 0x7AE5}, {0x7AEF, 0x7AEF}, {0x7AF6, 0x7AF6}, {0x7AF9, 0x7AF9}, {0x7B11, 0x7B11},
    {0x7B1B, 0x7B1B}, {0x7B26, 0x7B26}, {0x7B28, 0x7B28}, {0x7B2C, 0x7B2C}, {0x7B46, 0x7B46}, {0x7B49, 0x7B49},
    {0x7B4B, 0x7B4B}, {0x7B54, 0x7B54}, {0x7B56, 0x7B56}, {0x7B97, 0x7B97}, {0x7BA1, 0x7BA1}, {0x7BAD, 0x7BAD},
    {0x7BB1, 0x7BB1}, {0x7BC0, 0x7BC0}, {0x7BC4, 0x7BC4}, {0x7BC7, 0x7BC7}, {0x7BC9, 0x7BC9}, {0x7C21, 0x7C21},
    {0x7C2B, 0x7C2B}, {0x7C3D, 0x7C3D}, {0x7C3F, 0x7C3F}, {0x7C43, 0x7C43}, {0x7C4C, 0x7C4D}, {0x7C64, 0x7C64},
    {0x7C73, 0x7C73}, {0x7C89, 0x7C89}, {0x7C97, 0x7C97}, {0x7CB5, 0x7CB5}, {0x7CBE, 0x7CBE}, {0x7CCA, 0x7CCA},
    {0x7CD5, 0x7CD5}, {0x7CDF, 0x7CDF}, {0x7CF8, 0x7CF8}, {0x7CFB, 0x7CFB}, {0x7CFE, 0x7CFE}, {0x7D00, 0x7D00},
    {0x7D04, 0x7D05}, {0x7D0D, 0x7D0D}, {0x7D10, 0x7D10}, {0x7D14, 0x7D14}, {0x7D19, 0x7D1B}, {0x7D20, 0x7D20},
    {0x7D22, 0x7D22}, {0x7D2B, 0x7D2B}, {0x7D2F, 0x7D30}, {0x7D39, 0x7D39}, {0x7D42, 0x7D42}, {0x7D44, 0x7D44},
    {0x7D50, 0x7D50}, {0x7D55, 0x7D55}, {0x7D61, 0x7D61}, {0x7D66, 0x7D66}, {0x7D71, 0x7D72}, {0x7D93, 0x7D93},
    {0x7D9C, 0x7D9C}, {0x7DA0, 0x7DA0}, {0x7DAD, 0x7DAD}, {0x7DB1, 0x7DB2}, {0x7DCA, 0x7DCA}, {0x7DD2, 0x7DD2},
    {0x7DDA, 0x7DDA}, {0x7DE3, 0x7DE3}, {0x7DE8, 0x7DE9}, {0x7DEC, 0x7DEC}, {0x7DEF, 0x7DEF}, {0x7DF4, 0x7DF4},
    {0x7E1B, 0x7E1B}, {0x7E23, 0x7E23}, {0x7E2E, 0x7E2E}, {0x7E31, 0x7E31}, {0x7E3D, 0x7E3E}, {0x7E41, 0x7E41},
    {0x7E46, 0x7E46}, {0x7E54, 0x7E54}, {0x7E5E, 0x7E5E}, {0x7E6A, 0x7E6A}, {0x7E73, 0x7E73}, {0x7E7C, 0x7E7C},
    {0x7E8C, 0x7E8C}, {0x7F36, 0x7F36}, {0x7F38, 0x7F38}, {0x7F3A, 0x7F3A}, {0x7F51, 0x7F51}, {0x7F55, 0x7F55},
    {0x7F6A, 0x7F6A}, {0x7F6E, 0x7F6E}, {0x7F70, 0x7F70}, {0x7F72, 0x7F72}, {0x7F75, 0x7F75}, {0x7F77, 0x7F77},
    {0x7F85, 0x7F85}, {0x7F8A, 0x7F8A}, {0x7F8E, 0x7F8E}, {0x7F9E, 0x7F9E}, {0x7FA4, 0x7FA4}, {0x7FA9, 0x7FA9},
    {0x7FBD, 0x7FBD}, {0x7FC1, 0x7FC1}, {0x7FD2, 0x7FD2}, {0x7FD4, 0x7FD4}, {0x7FF0, 0x7FF0}, {0x7FF9, 0x7FF9},
    {0x7FFB, 0x7FFC}, {0x8000, 0x8001}, {0x8003, 0x8003}, {0x8005, 0x8005}, {0x800C, 0x800D}, {0x8010, 0x8010},
    {0x8012, 0x8012}, {0x8017, 0x8017}, {0x8033, 0x8033}, {0x8036, 0x8036}, {0x804A, 0x804A}, {0x8056, 0x8056},
    {0x805A, 0x805A}, {0x805E, 0x805E}, {0x806F, 0x8070}, {0x8072, 0x8072}, {0x8077, 0x8077}, {0x807D, 0x807D},
    {0x807F, 0x807F}, {0x8089, 0x8089}, {0x809A, 0x809A}, {0x80A1, 0x80A1}, {0x80A5, 0x80A5}, {0x80A9, 0x80A9},
    {0x80AF, 0x80AF}, {0x80B2, 0x80B2}, {0x80CC, 0x80CC}, {0x80CE, 0x80CE}, {0x80D6, 0x80D6}, {0x80DE, 0x80DE},
    {0x80E1, 0x80E1}, {0x80F8, 0x80F8}, {0x80FD, 0x80FD}, {0x8106, 0x8106}, {0x812B, 0x812B}, {0x8153, 0x8154},
    {0x8166, 0x8166}, {0x8170, 0x8170}, {0x8173, 0x8173}, {0x817F, 0x817F}, {0x81BD, 0x81BD}, {0x81C9, 0x81C9},
    {0x81D8, 0x81D8}, {0x81E3, 0x81E3}, {0x81E5, 0x81E5}, {0x81E8, 0x81E8}, {0x81EA, 0x81EA}, {0x81ED, 0x81ED},
    {0x81F3, 0x81F4}, {0x81FA, 0x81FA}, {0x81FC, 0x81FC}, {0x8207, 0x820A}, {0x820C, 0x820D}, {0x8212, 0x8212},
    {0x821B, 0x821B}, {0x821E, 0x821F}, {0x822A, 0x822A}, {0x822C, 0x822C}, {0x8239, 0x8239}, {0x8266, 0x8266},
    {0x826E, 0x826F}, {0x8272, 0x8272}, {0x8278, 0x8278}, {0x827E, 0x827E}, {0x829D, 0x829D}, {0x82AC, 0x82AC},
    {0x82B1, 0x82B1}, {0x82B3, 0x82B3}, {0x82E5, 0x82E6}, {0x82F1, 0x82F1}, {0x8305, 0x8305}, {0x832B, 0x832B},
    {0x8332, 0x8332}, {0x8336, 0x8336}, {0x8349, 0x8349}, {0x8352, 0x8352}, {0x8377, 0x8377}, {0x837C, 0x837C},
    {0x8389, 0x838A}, {0x838E, 0x838E}, {0x83AB, 0x83AB}, {0x83DC, 0x83DC}, {0x83E9, 0x83E9}, {0x83EF, 0x83EF},
    {0x83F2, 0x83F2}, {0x8404, 0x8404}, {0x840A, 0x840A}, {0x842C, 0x842C}, {0x843D, 0x843D}, {0x8449, 0x8449},
    {0x8457, 0x8457}, {0x845B, 0x845B}, {0x8461, 0x8461}, {0x8482, 0x8482}, {0x8499, 0x8499}, {0x84B2, 0x84B2},
    {0x84BC, 0x84BC}, {0x84CB, 0x84CB}, {0x84EE, 0x84EE}, {0x8515, 0x8515}, {0x8521, 0x8521}, {0x8523, 0x8523},
    {0x856D, 0x856D}, {0x8584, 0x8584}, {0x85A6, 0x85A6}, {0x85A9, 0x85AA}, {0x85C9, 0x85C9}, {0x85CD, 0x85CD},
    {0x85CF, 0x85CF}, {0x85DD, 0x85DD}, {0x85E4, 0x85E5}, {0x8606, 0x8607}, {0x862D, 0x862D}, {0x864D, 0x864E},
    {0x8655, 0x8655}, {0x865B, 0x865B}, {0x865F, 0x865F}, {0x8667, 0x8667}, {0x866B, 0x866B}, {0x86C7, 0x86C7},
    {0x86CB, 0x86CB}, {0x86D9, 0x86D9}, {0x8702, 0x8702}, {0x871C, 0x871C}, {0x8776, 0x8776}, {0x878D, 0x878D},
    {0x87A2, 0x87A2}, {0x87F2, 0x87F2}, {0x87F9, 0x87F9}, {0x880D, 0x880D}, {0x883B, 0x883B}, {0x8840, 0x8840},
    {0x884C, 0x884C}, {0x8853, 0x8853}, {0x8857, 0x8857}, {0x885B, 0x885B}, {0x885D, 0x885D}, {0x8861, 0x8861},
    {0x8863, 0x8863}, {0x8868, 0x8868}, {0x888B, 0x888B}, {0x88AB, 0x88AB}, {0x88C1, 0x88C2}, {0x88D5, 0x88D5},
    {0x88DC, 0x88DD}, {0x88E1, 0x88E1}, {0x88FD, 0x88FD}, {0x8907, 0x8907}, {0x8932, 0x8932}, {0x897E, 0x897F},
    {0x8981, 0x8981}, {0x8986, 0x8986}, {0x898B, 0x898B}, {0x898F, 0x898F}, {0x8996, 0x8996}, {0x89AA, 0x89AA},
    {0x89BA, 0x89BA}, {0x89BD, 0x89BD}, {0x89C0, 0x89C0}, {0x89D2, 0x89D2}, {0x89E3, 0x89E3}, {0x89F8, 0x89F8},
    {0x8A00, 0x8A00}, {0x8A02, 0x8A02}, {0x8A08, 0x8A08}, {0x8A0A, 0x8A0A}, {0x8A0E, 0x8A0E}, {0x8A13, 0x8A13},
    {0x8A17, 0x8A18}, {0x8A25, 0x8A25}, {0x8A2A, 0x8A2A}, {0x8A2D, 0x8A2D}, {0x8A31, 0x8A31}, {0x8A34, 0x8A34},
    {0x8A3B, 0x8A3C}, {0x8A55, 0x8A55}, {0x8A5E, 0x8A5E}, {0x8A62, 0x8A62}, {0x8A66, 0x8A66}, {0x8A69, 0x8A69},
    {0x8A71, 0x8A73}, {0x8A87, 0x8A87}, {0x8A8C, 0x8A8D}, {0x8A93, 0x8A93}, {0x8A95, 0x8A95}, {0x8A9E, 0x8A9E},
    {0x8AA0, 0x8AA0}, {0x8AA4, 0x8AA4}, {0x8AAA, 0x8AAA}, {0x8AB0, 0x8AB0}, {0x8AB2, 0x8AB2}, {0x8ABC, 0x8ABC},
    {0x8ABF, 0x8ABF}, {0x8AC7, 0x8AC7}, {0x8ACB, 0x8ACB}, {0x8AD2, 0x8AD2}, {0x8AD6, 0x8AD6}, {0x8AF8, 0x8AF8},
    {0x8AFA, 0x8AFA}, {0x8AFE, 0x8AFE}, {0x8B00, 0x8B00}, {0x8B02, 0x8B02}, {0x8B1B, 0x8B1B}, {0x8B1D, 0x8B1D},
    {0x8B49, 0x8B49}, {0x8B58, 0x8B58}, {0x8B5C, 0x8B5C}, {0x8B66, 0x8B66}, {0x8B6F, 0x8B70}, {0x8B77, 0x8B77},
    {0x8B7D, 0x8B7D}, {0x8B80, 0x8B80}, {0x8B8A, 0x8B8A}, {0x8B93, 0x8B93}, {0x8B9A, 0x8B9A}, {0x8C37, 0x8C37},
    {0x8C46, 0x8C46}, {0x8C48, 0x8C48}, {0x8C50, 0x8C50}, {0x8C55, 0x8C55}, {0x8C61, 0x8C61}, {0x8C6A, 0x8C6A},
    {0x8C6C, 0x8C6C}, {0x8C78, 0x8C78}, {0x8C8C, 0x8C8C}, {0x8C93, 0x8C93}, {0x8C9D, 0x8C9E}, {0x8CA0, 0x8CA2},
    {0x8CA8, 0x8CA8}, {0x8CAA, 0x8CAC}, {0x8CB4, 0x8CB4}, {0x8CB7, 0x8CB7}, {0x8CBB, 0x8CBC}, {0x8CC0, 0x8CC0},
    {0x8CC7, 0x8CC8}, {0x8CD3, 0x8CD3}, {0x8CDC, 0x8CDC}, {0x8CDE, 0x8CDE}, {0x8CE2, 0x8CE4}, {0x8CE6, 0x8CE6},
    {0x8CEA, 0x8CEA}, {0x8CED, 0x8CED}, {0x8CF4, 0x8CF4}, {0x8CFA, 0x8CFA}, {0x8CFC, 0x8CFD}, {0x8D08, 0x8D08},
    {0x8D0A, 0x8D0A}, {0x8D0F, 0x8D0F}, {0x8D64, 0x8D64}, {0x8D6B, 0x8D6B}, {0x8D70, 0x8D70}, {0x8D77, 0x8D77},
    {0x8D85, 0x8D85}, {0x8D8A, 0x8D8A}, {0x8D95, 0x8D95}, {0x8D99, 0x8D99}, {0x8DA3, 0x8DA3}, {0x8DA8, 0x8DA8},
    {0x8DB3, 0x8DB3}, {0x8DCC, 0x8DCC}, {0x8DCE, 0x8DCE}, {0x8DD1, 0x8DD1}, {0x8DDD, 0x8DDD}, {0x8DDF, 0x8DDF},
    {0x8DE1, 0x8DE1}, {0x8DEF, 0x8DEF}, {0x8DF3, 0x8DF3}, {0x8E0F, 0x8E0F}, {0x8E22, 0x8E22}, {0x8E5F, 0x8E5F},
    {0x8E64, 0x8E64}, {0x8E8D, 0x8E8D}, {0x8EAB, 0x8EAB}, {0x8EB2, 0x8EB2}, {0x8ECA, 0x8ECA}, {0x8ECC, 0x8ECD},
    {0x8ED2, 0x8ED2}, {0x8EDF, 0x8EDF}, {0x8F03, 0x8F03}, {0x8F09, 0x8F09}, {0x8F14, 0x8F15}, {0x8F1B, 0x8F1B},
    {0x8F1D, 0x8F1D}, {0x8F29, 0x8F2A}, {0x8F2F, 0x8F2F}, {0x8F38, 0x8F38}, {0x8F49, 0x8F49}, {0x8F5F, 0x8F5F},
    {0x8F9B, 0x8F9B}, {0x8FA6, 0x8FA6}, {0x8FA8, 0x8FA8}, {0x8FAD, 0x8FAD}, {0x8FAF, 0x8FB2}, {0x8FB5, 0x8FB5},
    {0x8FC5, 0x8FC5}, {0x8FCE, 0x8FCE}, {0x8FD1, 0x8FD1}, {0x8FD4, 0x8FD4}, {0x8FE6, 0x8FE6}, {0x8FEA, 0x8FEB},
    {0x8FF0, 0x8FF0}, {0x8FF4, 0x8FF4}, {0x8FF7, 0x8FF7}, {0x8FFD, 0x8FFD}, {0x9000, 0x9001}, {0x9003, 0x9003},
    {0x9006, 0x9006}, {0x900F, 0x9010}, {0x9014, 0x9014}, {0x9019, 0x901B}, {0x901D, 0x901D}, {0x901F, 0x9020},
    {0x9022, 0x9023}, {0x9031, 0x9032}, {0x9038, 0x9038}, {0x903C, 0x903C}, {0x9047, 0x9047}, {0x904A, 0x904B},
    {0x904D, 0x904E}, {0x9053, 0x9055}, {0x9059, 0x9059}, {0x905C, 0x905C}, {0x9060, 0x9060}, {0x9069, 0x9069},
    {0x906D, 0x906E}, {0x9072, 0x9072}, {0x9077, 0x9078}, {0x907A, 0x907A}, {0x907F, 0x9081}, {0x9084, 0x9084},
    {0x908A, 0x908A}, {0x908F, 0x908F}, {0x9091, 0x9091}, {0x90A3, 0x90A3}, {0x90A6, 0x90A6}, {0x90AA, 0x90AA},
    {0x90B1, 0x90B1}, {0x90CE, 0x90CE}, {0x90E8, 0x90E8}, {0x90ED, 0x90ED}, {0x90F5, 0x90F5}, {0x90FD, 0x90FD},
    {0x9102, 0x9102}, {0x9109, 0x9109}, {0x912D, 0x912D}, {0x9130, 0x9130}, {0x9149, 0x9149}, {0x914D, 0x914D},
    {0x9152, 0x9152}, {0x9177, 0x9178}, {0x9189, 0x9189}, {0x9192, 0x9192}, {0x919C, 0x919C}, {0x91AB, 0x91AB},
    {0x91C6, 0x91C7}, {0x91CB, 0x91CF}, {0x91D1, 0x91D1}, {0x91DD, 0x91DD}, {0x91E3, 0x91E3}, {0x9234, 0x9234},
    {0x9262, 0x9262}, {0x9280, 0x9280}, {0x9285, 0x9285}, {0x9296, 0x9296}, {0x9298, 0x9298}, {0x92B3, 0x92B3},
    {0x92B7, 0x92B7}, {0x92D2, 0x92D2}, {0x92FC, 0x92FC}, {0x9304, 0x9304}, {0x9322, 0x9322}, {0x9326, 0x9326},
    {0x932B, 0x932B}, {0x932F, 0x932F}, {0x934B, 0x934B}, {0x9375, 0x9375}, {0x937E, 0x937E}, {0x938A, 0x938A},
    {0x9396, 0x9396}, {0x93AE, 0x93AE}, {0x93E1, 0x93E1}, {0x9418, 0x9418}, {0x9435, 0x9435}, {0x9451, 0x9451},
    {0x9577, 0x9577}, {0x9580, 0x9580}, {0x9583, 0x9583}, {0x9589, 0x9589}, {0x958B, 0x958B}, {0x958F, 0x958F},
    {0x9592, 0x9593}, {0x95A3, 0x95A3}, {0x95B1, 0x95B1}, {0x95C6, 0x95C6}, {0x95CA, 0x95CA}, {0x95CD, 0x95CD},
    {0x95D0, 0x95D0}, {0x95DC, 0x95DC}, {0x95E1, 0x95E1}, {0x961C, 0x961C}, {0x9632, 0x9632}, {0x963B, 0x963B},
    {0x963F, 0x9640}, {0x9644, 0x9644}, {0x964D, 0x964D}, {0x9650, 0x9650}, {0x9662, 0x9664}, {0x966A, 0x966A},
    {0x9670, 0x9670}, {0x9673, 0x9673}, {0x9675, 0x9678}, {0x967D, 0x967D}, {0x9686, 0x9686}, {0x968A, 0x968A},
    {0x968E, 0x968E}, {0x9694, 0x9694}, {0x969B, 0x969C}, {0x96A8, 0x96A8}, {0x96AA, 0x96AA}, {0x96B1, 0x96B1},
    {0x96B6, 0x96B6}, {0x96B9, 0x96B9}, {0x96BB, 0x96BB}, {0x96C4, 0x96C6}, {0x96C9, 0x96C9}, {0x96D6, 0x96D6},
    {0x96D9, 0x96D9}, {0x96DC, 0x96DC}, {0x96DE, 0x96DE}, {0x96E2, 0x96E3}, {0x96E8, 0x96E8}, {0x96EA, 0x96EA},
    {0x96F2, 0x96F2}, {0x96F6, 0x96F7}, {0x96FB, 0x96FB}, {0x9700, 0x9700}, {0x9707, 0x9707}, {0x970D, 0x970D},
    {0x9727, 0x9727}, {0x9732, 0x9732}, {0x9738, 0x9739}, {0x9742, 0x9742}, {0x9748, 0x9748}, {0x9751, 0x9752},
    {0x9756, 0x9756}, {0x975C, 0x975C}, {0x975E, 0x975E}, {0x9760, 0x9760}, {0x9762, 0x9762}, {0x9769, 0x9769},
    {0x977C, 0x977C}, {0x978B, 0x978B}, {0x97C3, 0x97C3}, {0x97CB, 0x97CB}, {0x97D3, 0x97D3}, {0x97ED, 0x97ED},
    {0x97F3, 0x97F3}, {0x97FB, 0x97FB}, {0x97FF, 0x97FF}, {0x9801, 0x9802}, {0x9805, 0x9806}, {0x9808, 0x9808},
    {0x9810, 0x9811}, {0x9813, 0x9813}, {0x9817, 0x9818}, {0x981E, 0x981E}, {0x982D, 0x982D}, {0x983B, 0x983B},
    {0x9846, 0x9846}, {0x984C, 0x984D}, {0x984F, 0x984F}, {0x9858, 0x9858}, {0x985E, 0x985E}, {0x9867, 0x9867},
    {0x986F, 0x986F}, {0x98A8, 0x98A8}, {0x98C4, 0x98C4}, {0x98DB, 0x98DB}, {0x98DF, 0x98DF}, {0x98EF, 0x98EF},
    {0x98F2, 0x98F2}, {0x98FD, 0x98FE}, {0x9905, 0x9905}, {0x990A, 0x990A}, {0x9910, 0x9910}, {0x9918, 0x9918},
    {0x9928, 0x9928}, {0x9996, 0x9996}, {0x9999, 0x9999}, {0x99AC, 0x99AC}, {0x99D0, 0x99D0}, {0x99D5, 0x99D5},
    {0x99DB, 0x99DB}, {0x9A0E, 0x9A0E}, {0x9A19, 0x9A19}, {0x9A37, 0x9A37}, {0x9A45, 0x9A45}, {0x9A57, 0x9A57},
    {0x9A5A, 0x9A5A}, {0x9AA8, 0x9AA8}, {0x9AD4, 0x9AD4}, {0x9AD8, 0x9AD8}, {0x9ADF, 0x9ADF}, {0x9AEE, 0x9AEE},
    {0x9B06, 0x9B06}, {0x9B25, 0x9B25}, {0x9B27, 0x9B27}, {0x9B2F, 0x9B2F}, {0x9B31, 0x9B32}, {0x9B3C, 0x9B3C},
    {0x9B41, 0x9B42}, {0x9B45, 0x9B45}, {0x9B54, 0x9B54}, {0x9B5A, 0x9B5A}, {0x9B6F, 0x9B6F}, {0x9BAE, 0x9BAE},
    {0x9CE5, 0x9CE5}, {0x9CF3, 0x9CF4}, {0x9D3B, 0x9D3B}, {0x9D5D, 0x9D5D}, {0x9DF9, 0x9DF9}, {0x9E75, 0x9E75},
    {0x9E7F, 0x9E7F}, {0x9E97, 0x9E97}, {0x9EA5, 0x9EA5}, {0x9EB5, 0x9EB5}, {0x9EBB, 0x9EBC}, {0x9EC3, 0x9EC3},
    {0x9ECD, 0x9ECE}, {0x9ED1, 0x9ED1}, {0x9ED8, 0x9ED8}, {0x9EDE, 0x9EDE}, {0x9EE8, 0x9EE8}, {0x9EF9, 0x9EF9},
    {0x9EFD, 0x9EFD}, {0x9F0E, 0x9F0E}, {0x9F13, 0x9F13}, {0x9F20, 0x9F20}, {0x9F3B, 0x9F3B}, {0x9F4A, 0x9F4B},
    {0x9F52, 0x9F52}, {0x9F61, 0x9F61}, {0x9F8D, 0x9F8D}, {0x9F9C, 0x9F9C}, {0x9FA0, 0x9FA0}, {0xFE30, 0xFE44},
    {0xFE49, 0xFE52}, {0xFE54, 0xFE61}, {0xFE63, 0xFE63}, {0xFE68, 0xFE68}, {0xFE6A, 0xFE6B}, {0xFF01, 0xFF03},
    {0xFF05, 0xFF0A}, {0xFF0C, 0xFF0F}, {0xFF1A, 0xFF1B}, {0xFF1F, 0xFF20}, {0xFF3B, 0xFF3D}, {0xFF3F, 0xFF3F},
    {0xFF5B, 0xFF5B}, {0xFF5D, 0xFF5D},
}};
} // namespace

//---------------------------------------------------------------------------------------------------------------------
auto LatinWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint > 0x0000 && codePoint <= 0x007F)      // Basic Latin
           || (codePoint >= 0x0080 && codePoint <= 0x00FF)  // Latin-1 Supplement
           || (codePoint >= 0x0100 && codePoint <= 0x017F)  // Latin Extended-A
           || (codePoint >= 0x0180 && codePoint <= 0x024F)  // Latin Extended-B
           || (codePoint >= 0x0250 && codePoint <= 0x02AF)  // IPA Extensions
           || (codePoint >= 0x02B0 && codePoint <= 0x02FF)  // Spacing Modifier Letters
           || (codePoint >= 0x1D00 && codePoint <= 0x1D7F)  // Phonetic Extensions
           || (codePoint >= 0x1D80 && codePoint <= 0x1DBF)  // Phonetic Extensions Supplement
           || (codePoint >= 0x1E00 && codePoint <= 0x1EFF)  // Latin Extended Additional
           || (codePoint >= 0x2070 && codePoint <= 0x209F)  // Superscripts and Subscripts
           || (codePoint >= 0x2100 && codePoint <= 0x214F)  // Letterlike Symbols
           || (codePoint >= 0x2150 && codePoint <= 0x218F)  // Number Forms
           || (codePoint >= 0x2C60 && codePoint <= 0x2C7F)  // Latin Extended-C
           || (codePoint >= 0xA720 && codePoint <= 0xA7FF)  // Latin Extended-D
           || (codePoint >= 0xAB30 && codePoint <= 0xAB6F)  // Latin Extended-E
           || (codePoint >= 0xFB00 && codePoint <= 0xFB4F)  // Alphabetic Presentation Forms (Latin ligatures)
           || (codePoint >= 0xFF00 && codePoint <= 0xFFEF); // Halfwidth and Fullwidth Forms
}

//---------------------------------------------------------------------------------------------------------------------
auto GreekWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0370 && codePoint <= 0x03FF)     // Greek and Coptic
           || (codePoint >= 0x1F00 && codePoint <= 0x1FFF); // Greek Extended
}

//---------------------------------------------------------------------------------------------------------------------
auto CyrillicWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0400 && codePoint <= 0x04FF)     // Cyrillic
           || (codePoint >= 0x0500 && codePoint <= 0x052F)  // Cyrillic Supplement
           || (codePoint >= 0x2DE0 && codePoint <= 0x2DFF)  // Cyrillic Extended-A
           || (codePoint >= 0xA640 && codePoint <= 0xA69F)  // Cyrillic Extended-B
           || (codePoint >= 0x1C80 && codePoint <= 0x1C8F); // Cyrillic Extended-C
}

//---------------------------------------------------------------------------------------------------------------------
auto ArmenianWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0530 && codePoint <= 0x058F)     // Armenian
           || (codePoint >= 0xFB00 && codePoint <= 0xFB17); // Alphabetic Pres. Forms
}

//---------------------------------------------------------------------------------------------------------------------
auto HebrewWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0590 && codePoint <= 0x05FF)     // Hebrew,
           || (codePoint >= 0xFB1D && codePoint <= 0xFB4F); // Alphabetic Presentation Forms
}

//---------------------------------------------------------------------------------------------------------------------
auto ArabicWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0600 && codePoint <= 0x06FF)     // Arabic
           || (codePoint >= 0x0750 && codePoint <= 0x077F)  // Arabic Supplement
           || (codePoint >= 0x08A0 && codePoint <= 0x08FF)  // Arabic Extended-A
           || (codePoint >= 0x0870 && codePoint <= 0x089F)  // Arabic Extended-B
           || (codePoint >= 0xFB50 && codePoint <= 0xFDFF)  // Arabic Pres. Forms-A
           || (codePoint >= 0xFE70 && codePoint <= 0xFEFF); // Arabic Pres. Forms-B
}

//---------------------------------------------------------------------------------------------------------------------
auto SyriacWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0700 && codePoint <= 0x074F)     // Syriac
           || (codePoint >= 0x0860 && codePoint <= 0x086F); // Syriac Supplement
}

//---------------------------------------------------------------------------------------------------------------------
auto ThaanaWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0780 && codePoint <= 0x07BF; // Thaana
}

//---------------------------------------------------------------------------------------------------------------------
auto DevanagariWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0900 && codePoint <= 0x097F)     // Devanagari,
           || (codePoint >= 0xA8E0 && codePoint <= 0xA8FF)  // Devanagari Extended,
           || (codePoint >= 0x1CD0 && codePoint <= 0x1CFF); // Vedic Extensions
}

//---------------------------------------------------------------------------------------------------------------------
auto BengaliWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0980 && codePoint <= 0x09FF; // Bengali
}

//---------------------------------------------------------------------------------------------------------------------
auto GurmukhiWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0A00 && codePoint <= 0x0A7F; // Gurmukhi
}

//---------------------------------------------------------------------------------------------------------------------
auto GujaratiWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0A80 && codePoint <= 0x0AFF; // Gujarati
}

//---------------------------------------------------------------------------------------------------------------------
auto OriyaWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0B00 && codePoint <= 0x0B7F; // Oriya
}

//---------------------------------------------------------------------------------------------------------------------
auto TamilWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0B80 && codePoint <= 0x0BFF; // Tamil
}

//---------------------------------------------------------------------------------------------------------------------
auto TeluguWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0C00 && codePoint <= 0x0C7F; // Telugu
}

//---------------------------------------------------------------------------------------------------------------------
auto KannadaWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0C80 && codePoint <= 0x0CFF; // Kannada
}

//---------------------------------------------------------------------------------------------------------------------
auto MalayalamWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0D00 && codePoint <= 0x0D7F; // Malayalam
}

//---------------------------------------------------------------------------------------------------------------------
auto SinhalaWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0D80 && codePoint <= 0x0DFF; // Sinhala
}

//---------------------------------------------------------------------------------------------------------------------
auto ThaiWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0E00 && codePoint <= 0x0E7F; // Thai
}

//---------------------------------------------------------------------------------------------------------------------
auto LaoWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0E80 && codePoint <= 0x0EFF; // Lao
}

//---------------------------------------------------------------------------------------------------------------------
auto TibetanWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x0F00 && codePoint <= 0x0FFF; // Final Accepted Script Proposal of the First Usable Edition
}

//---------------------------------------------------------------------------------------------------------------------
auto MyanmarWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x1000 && codePoint <= 0x109F)     // Myanmar
           || (codePoint >= 0xAA60 && codePoint <= 0xAA7F)  // Myanmar Extended-A
           || (codePoint >= 0xA9E0 && codePoint <= 0xA9FF); // Myanmar Extended-B
}

//---------------------------------------------------------------------------------------------------------------------
auto GeorgianWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x10A0 && codePoint <= 0x10FF)     // Georgian
           || (codePoint >= 0x2D00 && codePoint <= 0x2D2F)  // Supplement
           || (codePoint >= 0x1C90 && codePoint <= 0x1CBF); // Extended
}

//---------------------------------------------------------------------------------------------------------------------
auto KhmerWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x1780 && codePoint <= 0x17FF)     // Khmer
           || (codePoint >= 0x19E0 && codePoint <= 0x19FF); // Khmer Symbols
}

//---------------------------------------------------------------------------------------------------------------------
auto SimplifiedChineseWritingSystem(QChar c) -> bool
{
    char16_t codePoint = c.unicode();

    auto it = std::lower_bound(simplifiedChineseWritingSystemRanges.begin(),
                               simplifiedChineseWritingSystemRanges.end(),
                               codePoint,
                               [](const Range &range, char16_t value) { return range.end < value; });
    return it != simplifiedChineseWritingSystemRanges.end() && it->start <= codePoint && codePoint <= it->end;
}

//---------------------------------------------------------------------------------------------------------------------
auto TraditionalChineseWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    auto it = std::lower_bound(traditionalChineseWritingSystemRanges.begin(),
                               traditionalChineseWritingSystemRanges.end(),
                               codePoint,
                               [](const Range &range, char16_t value) { return range.end < value; });
    return it != traditionalChineseWritingSystemRanges.end() && it->start <= codePoint && codePoint <= it->end;
}

//---------------------------------------------------------------------------------------------------------------------
auto JapaneseWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x3040 && codePoint <= 0x309F)     // Hiragana
           || (codePoint >= 0x30A0 && codePoint <= 0x30FF)  // Katakana
           || (codePoint >= 0x4E00 && codePoint <= 0x9FBF); // Kanji
}

//---------------------------------------------------------------------------------------------------------------------
auto KoreanWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x1100 && codePoint <= 0x11FF) || (codePoint >= 0x3130 && codePoint <= 0x318F) ||
           (codePoint >= 0xA960 && codePoint <= 0xA97F) || (codePoint >= 0xD7B0 && codePoint <= 0xD7FF);
}

//---------------------------------------------------------------------------------------------------------------------
auto VietnameseWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x0020 && codePoint <= 0x002F) || (codePoint >= 0x0030 && codePoint <= 0x0039) ||
           (codePoint >= 0x003A && codePoint <= 0x0040) || (codePoint >= 0x0041 && codePoint <= 0x005A) ||
           (codePoint >= 0x005B && codePoint <= 0x0060) || (codePoint >= 0x0061 && codePoint <= 0x007A) ||
           (codePoint >= 0x007B && codePoint <= 0x007E) || (codePoint >= 0x00C0 && codePoint <= 0x00C3) ||
           (codePoint >= 0x00C8 && codePoint <= 0x00CA) || (codePoint >= 0x00CC && codePoint <= 0x00CD) ||
           codePoint == 0x00D0 || (codePoint >= 0x00D2 && codePoint <= 0x00D5) ||
           (codePoint >= 0x00D9 && codePoint <= 0x00DA) || codePoint == 0x00DD ||
           (codePoint >= 0x00E0 && codePoint <= 0x00E3) || (codePoint >= 0x00E8 && codePoint <= 0x00EA) ||
           (codePoint >= 0x00EC && codePoint <= 0x00ED) || (codePoint >= 0x00F2 && codePoint <= 0x00F5) ||
           (codePoint >= 0x00F9 && codePoint <= 0x00FA) || codePoint == 0x00FD ||
           (codePoint >= 0x0102 && codePoint <= 0x0103) || (codePoint >= 0x0110 && codePoint <= 0x0111) ||
           (codePoint >= 0x0128 && codePoint <= 0x0129) || (codePoint >= 0x0168 && codePoint <= 0x0169) ||
           (codePoint >= 0x01A0 && codePoint <= 0x01B0) || (codePoint >= 0x1EA0 && codePoint <= 0x1EF9) ||
           (codePoint >= 0x02C6 && codePoint <= 0x0323);
}

//---------------------------------------------------------------------------------------------------------------------
auto SymbolWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return (codePoint >= 0x2600 && codePoint <= 0x26FF)     // Miscellaneous Symbols
           || (codePoint >= 0x19E0 && codePoint <= 0x19FF)  // Khmer Symbols
           || (codePoint >= 0x20A0 && codePoint <= 0x20CF)  // Currency Symbols
           || (codePoint >= 0x20D0 && codePoint <= 0x20FF)  // Combining Diacritical Marks for Symbols
           || (codePoint >= 0x2100 && codePoint <= 0x214F)  // Letterlike Symbols
           || (codePoint >= 0x27C0 && codePoint <= 0x27EF)  // Miscellaneous Mathematical Symbols-A
           || (codePoint >= 0x2980 && codePoint <= 0x29FF)  // Miscellaneous Mathematical Symbols-B
           || (codePoint >= 0x2B00 && codePoint <= 0x2BFF)  // Miscellaneous Symbols and Arrows
           || (codePoint >= 0x3000 && codePoint <= 0x303F)  // CJK Symbols and Punctuation
           || (codePoint >= 0x4DC0 && codePoint <= 0x4DFF); // Yijing Hexagram Symbols
}

//---------------------------------------------------------------------------------------------------------------------
auto OghamWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x1680 && codePoint <= 0x169F; // Ogham
}

//---------------------------------------------------------------------------------------------------------------------
auto RunicWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x16A0 && codePoint <= 0x16FF; // Runic
}

//---------------------------------------------------------------------------------------------------------------------
auto NkoWritingSystem(QChar c) -> bool
{
    char16_t const codePoint = c.unicode();

    return codePoint >= 0x07C0 && codePoint <= 0x07FF; // Nko
}

//---------------------------------------------------------------------------------------------------------------------
auto WritingSystemSample(SVGFontWritingSystem writingSystem) -> QString
{
    return [&]() -> QStringView
    {
        switch (writingSystem)
        {
            case SVGFontWritingSystem::Any:
            case SVGFontWritingSystem::Symbol:
                // show only ascii characters
                return u"AaBbzZ";
            case SVGFontWritingSystem::Latin:
                return u"Aa\x00C3\x00E1Zz";
            case SVGFontWritingSystem::Greek:
                return u"\x0393\x03B1\x03A9\x03C9";
            case SVGFontWritingSystem::Cyrillic:
                return u"\x0414\x0434\x0436\x044f";
            case SVGFontWritingSystem::Armenian:
                return u"\x053f\x054f\x056f\x057f";
            case SVGFontWritingSystem::Hebrew:
                return u"\x05D0\x05D1\x05D2\x05D3";
            case SVGFontWritingSystem::Arabic:
                return u"\x0623\x0628\x062C\x062F\x064A\x0629\x0020\x0639\x0631\x0628\x064A\x0629";
            case SVGFontWritingSystem::Syriac:
                return u"\x0715\x0725\x0716\x0726";
            case SVGFontWritingSystem::Thaana:
                return u"\x0784\x0794\x078c\x078d";
            case SVGFontWritingSystem::Devanagari:
                return u"\x0905\x0915\x0925\x0935";
            case SVGFontWritingSystem::Bengali:
                return u"\x0986\x0996\x09a6\x09b6";
            case SVGFontWritingSystem::Gurmukhi:
                return u"\x0a05\x0a15\x0a25\x0a35";
            case SVGFontWritingSystem::Gujarati:
                return u"\x0a85\x0a95\x0aa5\x0ab5";
            case SVGFontWritingSystem::Oriya:
                return u"\x0b06\x0b16\x0b2b\x0b36";
            case SVGFontWritingSystem::Tamil:
                return u"\x0b89\x0b99\x0ba9\x0bb9";
            case SVGFontWritingSystem::Telugu:
                return u"\x0c05\x0c15\x0c25\x0c35";
            case SVGFontWritingSystem::Kannada:
                return u"\x0c85\x0c95\x0ca5\x0cb5";
            case SVGFontWritingSystem::Malayalam:
                return u"\x0d05\x0d15\x0d25\x0d35";
            case SVGFontWritingSystem::Sinhala:
                return u"\x0d90\x0da0\x0db0\x0dc0";
            case SVGFontWritingSystem::Thai:
                return u"\x0e02\x0e12\x0e22\x0e32";
            case SVGFontWritingSystem::Lao:
                return u"\x0e8d\x0e9d\x0ead\x0ebd";
            case SVGFontWritingSystem::Tibetan:
                return u"\x0f00\x0f01\x0f02\x0f03";
            case SVGFontWritingSystem::Myanmar:
                return u"\x1000\x1001\x1002\x1003";
            case SVGFontWritingSystem::Georgian:
                return u"\x10a0\x10b0\x10c0\x10d0";
            case SVGFontWritingSystem::Khmer:
                return u"\x1780\x1790\x17b0\x17c0";
            case SVGFontWritingSystem::SimplifiedChinese:
                return u"\x4e2d\x6587\x8303\x4f8b";
            case SVGFontWritingSystem::TraditionalChinese:
                return u"\x4e2d\x6587\x7bc4\x4f8b";
            case SVGFontWritingSystem::Japanese:
                return u"\x30b5\x30f3\x30d7\x30eb\x3067\x3059";
            case SVGFontWritingSystem::Korean:
                return u"\xac00\xac11\xac1a\xac2f";
            case SVGFontWritingSystem::Vietnamese:
                return u"\x1ED7\x1ED9\x1ED1\x1ED3";
            case SVGFontWritingSystem::Ogham:
                return u"\x1681\x1682\x1683\x1684";
            case SVGFontWritingSystem::Runic:
                return u"\x16a0\x16a1\x16a2\x16a3";
            case SVGFontWritingSystem::Nko:
                return u"\x7ca\x7cb\x7cc\x7cd";
            default:
                return nullptr;
        }
    }()
                        .toString();
}
