/*
 * Copyright (C) 2020 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(GPU_PROCESS) && ENABLE(WEBGL)

#include "Connection.h"
#include "GPUConnectionToWebProcess.h"
#include "GPUProcess.h"
#include "GraphicsContextGLIdentifier.h"
#include "RemoteRenderingBackend.h"
#include "RemoteSharedResourceCache.h"
#include "ScopedWebGLRenderingResourcesRequest.h"
#include "SharedPreferencesForWebProcess.h"
#include "SharedVideoFrame.h"
#include "StreamMessageReceiver.h"
#include "StreamServerConnection.h"
#include <WebCore/NotImplemented.h>
#include <WebCore/PixelBuffer.h>
#include <WebCore/ProcessIdentity.h>
#include <WebCore/RenderingResourceIdentifier.h>
#include <wtf/TZoneMalloc.h>
#include <wtf/ThreadAssertions.h>
#include <wtf/ThreadSafeWeakPtr.h>
#include <wtf/WeakPtr.h>

#if PLATFORM(COCOA)
#include <WebCore/GraphicsContextGLCocoa.h>
#elif USE(GBM)
#include <WebCore/DMABufBuffer.h>
#include <WebCore/GraphicsContextGLTextureMapperGBM.h>
#else
#include <WebCore/GraphicsContextGLTextureMapperANGLE.h>
#endif

#if PLATFORM(MAC)
#include <CoreGraphics/CGDisplayConfiguration.h>
#endif

#if USE(GRAPHICS_LAYER_WC)
#include "WCContentBufferIdentifier.h"
#endif

#if ENABLE(VIDEO)
#include "RemoteVideoFrameIdentifier.h"
#include "RemoteVideoFrameProxy.h"
#endif

#if PLATFORM(COCOA)
namespace WTF {
class MachSendRight;
}
#endif

namespace WebKit {
#if ENABLE(VIDEO)
class RemoteVideoFrameObjectHeap;
#endif

IPC::StreamConnectionWorkQueue& remoteGraphicsContextGLStreamWorkQueueSingleton();


// GPU process side implementation of that receives messages about GraphicsContextGL calls
// and issues real GraphicsContextGL calls based on the received messages.
// The implementation is largely generated by running Tools/Scripts/generate-gpup-webgl.
class RemoteGraphicsContextGL : private WebCore::GraphicsContextGL::Client, public IPC::StreamMessageReceiver {
    WTF_MAKE_TZONE_ALLOCATED(RemoteGraphicsContextGL);
public:
    static Ref<RemoteGraphicsContextGL> create(GPUConnectionToWebProcess&, WebCore::GraphicsContextGLAttributes&&, GraphicsContextGLIdentifier, RemoteRenderingBackend&, Ref<IPC::StreamServerConnection>&&);
    ~RemoteGraphicsContextGL() override;
    void stopListeningForIPC(Ref<RemoteGraphicsContextGL>&& refFromConnection);

    std::optional<SharedPreferencesForWebProcess> sharedPreferencesForWebProcess() const { return m_sharedPreferencesForWebProcess; }

    void didReceiveStreamMessage(IPC::StreamServerConnection&, IPC::Decoder&) final;

protected:
    RemoteGraphicsContextGL(GPUConnectionToWebProcess&, GraphicsContextGLIdentifier, RemoteRenderingBackend&, Ref<IPC::StreamServerConnection>&&);
    void initialize(WebCore::GraphicsContextGLAttributes&&);
    IPC::StreamConnectionWorkQueue& workQueue() const { return m_workQueue; }

    void workQueueInitialize(WebCore::GraphicsContextGLAttributes&&);
    virtual void platformWorkQueueInitialize(WebCore::GraphicsContextGLAttributes&&) { };
    void workQueueUninitialize();
    template<typename T>
    IPC::Error send(T&& message) const
    {
        return Ref { *m_streamConnection }->send(std::forward<T>(message), m_graphicsContextGLIdentifier);
    }

    // GraphicsContextGL::Client overrides.
    void forceContextLost() final;
    void addDebugMessage(GCGLenum, GCGLenum, GCGLenum, const String&) final;

    // Messages to be received.
    void ensureExtensionEnabled(String&&);
    void reshape(int32_t width, int32_t height);
#if PLATFORM(COCOA)
    virtual void prepareForDisplay(IPC::Semaphore&&, CompletionHandler<void(WTF::MachSendRight&&)>&&) = 0;
#elif USE(GRAPHICS_LAYER_WC)
    virtual void prepareForDisplay(CompletionHandler<void(std::optional<WCContentBufferIdentifier>)>&&) = 0;
#elif USE(GBM)
    virtual void prepareForDisplay(CompletionHandler<void(uint64_t, std::optional<WebCore::DMABufBuffer::Attributes>&&, UnixFileDescriptor&&)>&&) = 0;
#else
    void prepareForDisplay(CompletionHandler<void()>&&);
#endif
    void getErrors(CompletionHandler<void(GCGLErrorCodeSet)>&&);
    void drawSurfaceBufferToImageBuffer(WebCore::GraphicsContextGL::SurfaceBuffer, WebCore::RenderingResourceIdentifier, CompletionHandler<void()>&&);
#if ENABLE(MEDIA_STREAM) || ENABLE(WEB_CODECS)
    void surfaceBufferToVideoFrame(WebCore::GraphicsContextGL::SurfaceBuffer, CompletionHandler<void(std::optional<WebKit::RemoteVideoFrameProxy::Properties>&&)>&&);
#endif
#if ENABLE(VIDEO) && PLATFORM(COCOA)
    void copyTextureFromVideoFrame(SharedVideoFrame&&, PlatformGLObject texture, uint32_t target, int32_t level, uint32_t internalFormat, uint32_t format, uint32_t type, bool premultiplyAlpha, bool flipY, CompletionHandler<void(bool)>&&);
    void setSharedVideoFrameSemaphore(IPC::Semaphore&&);
    void setSharedVideoFrameMemory(WebCore::SharedMemory::Handle&&);
#endif
    void simulateEventForTesting(WebCore::GraphicsContextGL::SimulatedEventForTesting);
    void getBufferSubDataInline(uint32_t target, uint64_t offset, uint64_t dataSize, CompletionHandler<void(std::span<const uint8_t>)>&&);
    void getBufferSubDataSharedMemory(uint32_t target, uint64_t offset, uint64_t dataSize, WebCore::SharedMemory::Handle, CompletionHandler<void(bool)>&&);
    void readPixelsInline(WebCore::IntRect, uint32_t format, uint32_t type, bool packReverseRowOrder, CompletionHandler<void(std::optional<WebCore::IntSize>, std::span<const uint8_t>)>&&);
    void readPixelsSharedMemory(WebCore::IntRect, uint32_t format, uint32_t type, bool packReverseRowOrder, WebCore::SharedMemory::Handle, CompletionHandler<void(std::optional<WebCore::IntSize>)>&&);
    void multiDrawArraysANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t>&& firstsAndCounts);
    void multiDrawArraysInstancedANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t>&& firstsCountsAndInstanceCounts);
    void multiDrawElementsANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t>&& countsAndOffsets, uint32_t type);
    void multiDrawElementsInstancedANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t>&& countsOffsetsAndInstanceCounts, uint32_t type);
    void multiDrawArraysInstancedBaseInstanceANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t, uint32_t>&& firstsCountsInstanceCountsAndBaseInstances);
    void multiDrawElementsInstancedBaseVertexBaseInstanceANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t, int32_t, uint32_t>&& countsOffsetsInstanceCountsBaseVerticesAndBaseInstances, uint32_t type);
    void drawBuffers(std::span<const uint32_t>);
    void drawBuffersEXT(std::span<const uint32_t>);
    void invalidateFramebuffer(uint32_t target, std::span<const uint32_t> attachments);
    void invalidateSubFramebuffer(uint32_t target, std::span<const uint32_t> attachments, int32_t x, int32_t y, int32_t width, int32_t height);
#if ENABLE(WEBXR)
    void framebufferDiscard(uint32_t target, std::span<const uint32_t> attachments);
#endif

#if ENABLE(VIDEO)
    Ref<RemoteVideoFrameObjectHeap> protectedVideoFrameObjectHeap() const;
#endif

#if PLATFORM(COCOA)
    using GCGLContext = WebCore::GraphicsContextGLCocoa;
#elif USE(GBM)
    using GCGLContext = WebCore::GraphicsContextGLTextureMapperGBM;
#else
    using GCGLContext = WebCore::GraphicsContextGLTextureMapperANGLE;
#endif

#include "RemoteGraphicsContextGLFunctionsGenerated.h" // NOLINT

private:
    void paintNativeImageToImageBuffer(WebCore::NativeImage&, WebCore::RenderingResourceIdentifier);
    bool webXREnabled() const;
    bool webXRPromptAccepted() const;
    Ref<IPC::StreamConnectionWorkQueue> protectedWorkQueue() const { return m_workQueue; }
    RefPtr<GCGLContext> protectedContext();

protected:
    ThreadSafeWeakPtr<GPUConnectionToWebProcess> m_gpuConnectionToWebProcess;
    Ref<IPC::StreamConnectionWorkQueue> m_workQueue;
    RefPtr<IPC::StreamServerConnection> m_streamConnection;
    RefPtr<GCGLContext> m_context WTF_GUARDED_BY_CAPABILITY(workQueue());
    GraphicsContextGLIdentifier m_graphicsContextGLIdentifier;
    Ref<RemoteRenderingBackend> m_renderingBackend;
    Ref<RemoteSharedResourceCache> m_sharedResourceCache;
#if ENABLE(VIDEO)
    Ref<RemoteVideoFrameObjectHeap> m_videoFrameObjectHeap;
#if PLATFORM(COCOA)
    SharedVideoFrameReader m_sharedVideoFrameReader;
#endif
#endif
    ScopedWebGLRenderingResourcesRequest m_renderingResourcesRequest;
    WebCore::ProcessIdentifier m_webProcessIdentifier;
    SharedPreferencesForWebProcess m_sharedPreferencesForWebProcess;
    HashMap<uint32_t, PlatformGLObject, IntHash<uint32_t>, WTF::UnsignedWithZeroKeyHashTraits<uint32_t>> m_objectNames;
};


} // namespace WebKit

#endif
